/*
 * evsumdiv.c - mex file:    new file, implementing the division by Evsum
 *                           needed in solving the Lyapunov systems
 *
 * synopsis:   B = evsumdiv(A,Evsum,blk)
 *
 * inputs:
 *    A                 an nxn block diagonal symmetric matrix
 *    Evsum             an nxn block diagonal symmetric matrix
 *    blk               the block diagonal structure
 *
 * output:
 *    B                 an nxn block diagonal symmetric matrix
 *                      equal to A ./ Evsum
 *
 * Notes:
 *  1.   the blocks of Evsum are assumed to be full, i.e. it is the
 *       user's responsability to ensure that all entries in the blocks
 *       of Evsum are nonzero; this is reasonable, since otherwise
 *       division by 0 will occur; on the other hand, we can not assume
 *       that all the entries in the blocks of A are nonzero.
 *       (evsumdiv checks for nonzeros in Evsum hen the matrices are sparse,
 *       but not in the case of a single full block)
 *
 *  2.   in the case of a single full block A and Evsum must be symmetric
 *       to produce the right answer, namely A ./ Evsum; if the matrices
 *       are sparse, the answer is always equal to A ./ Evsum
 *
 * Copyright (c) 1997 by
 * F. Alizadeh, J.-P. Haeberly, M. Nayakkankuppam, M.L. Overton
 * Last modified : 3/8/97
 */
#include <math.h>
#include "mex.h"

/* Input Arguments */
#define  A_IN       prhs[0]
#define  Ev_IN      prhs[1]
#define  blk_IN     prhs[2]

/* Output Arguments */
#define  B_OUT  plhs[0]

#if !defined(max)
#define  max(A, B)   ((A) > (B) ? (A) : (B))
#endif

#if !defined(min)
#define  min(A, B)   ((A) < (B) ? (A) : (B))
#endif

static void evsumdiv(
   double  *bpr,
   int  *bir,
   int  *bjc,
   double  *apr,
   int  *air,
   int  *ajc,
   double  *evpr,
   int  *evir,
   int  *evjc,
   int  n
)
{
   int i,j,start,fin,idx,rowidx;
   double *aiter,*eviter,*biter;

   if (air != NULL) { /* matrices are sparse; note that B will have */
                      /* exactly the same sparse structure as A */
      memcpy(bjc,ajc,(n+1)*sizeof(int));
      memcpy(bir,air,ajc[n]*sizeof(int));
      start = ajc[0];
      for (j = 0; j < n; j++) {  /* loop over the columns */
         fin = ajc[j+1]; /* fin-1 is index in ir, pr of last nonzero entry in jth col */
         idx = evjc[j];  /* idx is index in evir, evpr of first nonzero entry in jth col */
         for (i = start; i < fin; i++) {
            rowidx = air[i];  /* retrieve row index */
            while (evir[idx] < rowidx) /* search for entry in evpr corresponding to */
               ++idx;                  /* jth col and row rowidx */
            bpr[i] = apr[i] / evpr[idx];
         }
         start = fin;
      }
   }
   else {          /* one full block only */
      biter = bpr;
      aiter = apr;
      eviter = evpr;
      for(i = n; i > 1; i--) {   /* loop over the columns */
         biter[0] = aiter[0] / eviter[0]; /* diagonal entry */
         idx = n;
         for(j = 1; j < i; j++) {   /* piece of column below the diagonal */
            biter[j] = aiter[j] / eviter[j];
            biter[idx] = biter[j];
            idx += n;
         }
         biter += n+1;
         aiter += n+1;
         eviter += n+1;
      }
      biter[0] = aiter[0] / eviter[0];
   }
   return;
}

void mexFunction(
   int nlhs,       mxArray *plhs[],
   int nrhs, const mxArray *prhs[]
)
{
   double *apr,*evpr,*bpr,*blk;
   int i,j,n,sumblk2,nblk;
   int *air,*ajc,*evir,*evjc,*bir,*bjc;
   bool is_sparse;

/* Check for proper number of arguments */
   if (nrhs != 3) {
      mexErrMsgTxt("evsumdiv requires three input arguments.");
   } else if (nlhs > 1) {
      mexErrMsgTxt("evsumdiv requires one output argument.");
   }

   i = mxGetM(A_IN);
   n = mxGetN(A_IN);
   if (i != n)
      mexErrMsgTxt("evsumdiv: A must be square.");
   j = mxGetM(Ev_IN);
   n = mxGetN(Ev_IN);
   if (j != n)
      mexErrMsgTxt("evsumdiv: Evsum must be square.");
   if (i != n)
      mexErrMsgTxt("evsumdiv: A and Evsum must have same dimension.");
   i = mxGetM(blk_IN);
   nblk = mxGetN(blk_IN);
   nblk = max(i,nblk);
   if (nblk > n)
      mexErrMsgTxt("evsumdiv: too many blocks.");

/* consistency check */
   is_sparse = mxIsSparse(A_IN);
   if ((nblk > 1) && !is_sparse)
      mexErrMsgTxt("evsumdiv: A should be sparse");
   if ((is_sparse && !mxIsSparse(Ev_IN)) || (!is_sparse && mxIsSparse(Ev_IN)))
      mexErrMsgTxt("evsumdiv: A and Evsum must be both full or both sparse");

/* Assign pointers to the various input parameters */
   apr = mxGetPr(A_IN);
   evpr = mxGetPr(Ev_IN);
   blk = mxGetPr(blk_IN);
   sumblk2 = 0;
   for(i = 0; i < nblk; i++) {
      j = blk[i];
      sumblk2 += j*j;
   }
   if (is_sparse) {
      air = mxGetIr(A_IN);
      ajc = mxGetJc(A_IN);
      evir = mxGetIr(Ev_IN);
      evjc = mxGetJc(Ev_IN);
      if (evjc[n] < sumblk2)
         mexErrMsgTxt("evsumdiv: error! Evsum has a zero entry.");
   } else {
      air = NULL;
      ajc = NULL;
      evir = NULL;
      evjc = NULL;
   }

/* Create a matrix for the return argument */
   if(is_sparse) {
      B_OUT = mxCreateSparse(n,n,sumblk2,mxREAL);
      bir = mxGetIr(B_OUT);
      bjc = mxGetJc(B_OUT);
   }
   else {
      B_OUT = mxCreateDoubleMatrix(n,n,mxREAL);
      bir = NULL;
      bjc = NULL;
   }

/* Assign pointers to the output parameter */
   bpr = mxGetPr(B_OUT);

/* Do the actual computations in a subroutine */
   evsumdiv(bpr,bir,bjc,apr,air,ajc,evpr,evir,evjc,n);
   return;
}
