function failed = export(fname,A,b,C,blk)
% EXPORT  exports SQL data from Matlab to an ASCII file using a
%         variant of Nemirovskii's compact format
%
% failed = export(fname,A,b,C,blk)
%
% Note: write one value per line so file can be loaded using
% Matlab's 'load' command.
%
% input variables:
%     fname:        the name of the output file
%     - A                  structure of constraints data
%     -     A.s            constraint matrix for SD
%     -     A.q            constraint matrix for QC
%     -     A.l            constraint matrix for LP
%     - b                  rhs of primal constraints
%     - C                  cost structure
%     -     C.s            cost matrix for SD
%     -     C.q            cost vector for QC
%     -     C.l            cost vector for LP
%     - blk                block info structure
%     -     blk.s          block info for SD
%     -     blk.q          block info for QC
%     -     blk.l          block info for LP
%
% output variables:
%     - failed    flag: 0 = success, 1 = failure

% The storage scheme is a follows:
%     -  m              # of constraints
%     -  b              the vector of length m, the right-hand-side of primal constraints
% next the SD data
%     -  nblk           # of SD blocks; 0 means that problem has no SD component
%     -  blk            block size structure for SD: vector of length nblk
%     ================ the matrix C.s
%     === loop over the blocks
%     -  spblk          0 = dense block; 1 = sparse block
%     ====== if block is sparse
%     -  nz = # of nonzeros in block
%     ========= loop over # nonzero entries
%     -  <row#> <col#> <value>  of entry
%     ====== else if block is dense
%     -  <value> for all entries in upper triangle ordered rowise
%     ================ loop over k = 1:m
%     ====== the matrix Ak using same scheme as for C.s
%     ====== loop over the blocks
%     -  spblk          0 = dense block; 1 = sparse block
%     ====== if block is sparse
%     -  # of nonzeros in block
%     ========= loop over # nonzero entries
%     -  <row#> <col#> <value>  of entry
%     ====== else if block is dense
%     -  <value> for all entries in upper triangle ordered rowise
%     ================
% next the QC data
%     -  nblk           # of QC blocks; 0 means that problem has no QC component
%     -  blk            block size structure for QC: vector of length nblk
%     ================
%     -  c              the vector determining the QC component of the objective function
%     ================  now the matrix A.q
%     - spblk           0 = dense matrix; 1 = sparse matrix
%     ====== if A.q is sparse
%     -  # of nonzeros in A.q
%     ========= loop over # nonzero entries
%     -  <row#> <col#> <value>  of entry
%     ====== else if A.q is dense
%     -  <value>        of all the entries of A.q columnwise
%     ================
% finally the LP data   same as QP data
%     -  nblk           # of LP blocks; 0 means that problem has no LP component
%     -  blk            block size structure for LP: vector of length nblk
%     ================
%     -  c              the vector determining the LP component of the objective function
%     ================  now the matrix A.l
%     - spblk           0 = dense matrix; 1 = sparse matrix
%     ====== if A.l is sparse
%     -  # of nonzeros in A.l
%     ========= loop over # nonzero entries
%     -  <row#> <col#> <value>  of entry
%     ====== else if A.l is dense
%     -  <value>        of all the entries of A.l columnwise
%     ================

% Copyright (c) 1997 by
% F. Alizadeh, J.-P. Haeberly, M. Nayakkankuppam, M.L. Overton, S. Schmieta
% Last modified: 5/11/97
%
 if nargin ~= 5
    fprintf('\nexport: not enough arguments! Aborting...\n');
    failed = 1;
    return
 end
%
 fp = fopen(fname,'wt');
 if fp == -1
    fprintf('\nexport: can not open file %s!  Aborting...\n',fname);
    failed = 1;
    return
 else
    failed = 0;
 end
%
% first write m and b
%
 m = length(b);
 fprintf(fp,'%d\n',m);            % m = # constraints
 fprintf(fp,'%20.16e\n',b);       % write the vector b
%
% next the SD data; recall that for the Matlab version of the code, the blocks
% are either all sparse (if the matrix constraint A.s is sparse) or all dense.
%
 if isfield(blk,'s')
    nblk = length(blk.s);
    if sum(blk.s) <= 0
       nblk = 0;
    end
 else
    nblk = 0;
 end
 fprintf(fp,'%d\n',nblk);        % nblk = # blocks
 if nblk > 0                     % if nblk = 0, no SD component to problem
    fprintf(fp,'%d\n',blk.s);    % write block structure
    if issparse(A.s)
       sparseblocks = 1;         % block sparsity structure: all sparse
    else
       sparseblocks = 0;         % all dense
    end
%
% make lookup table of starting points of blocks
%
    start(1) = 0;
    for k = 2:length(blk.s)
       start(k) = start(k-1) + blk.s(k-1);
    end;
%
% write the matrix C.s
%
    for blkidx = 1:nblk,
       bsize = blk.s(blkidx);
       ulc = start(blkidx);
       lrc = ulc + bsize;
       ulc = ulc+1;
       tmpmat = C.s(ulc:lrc,ulc:lrc);
       if sparseblocks == 1,
          fprintf(fp,'%d\n',1);
%
% find row/columns indices and values of nonzero entries of block
% note that the entries are ordered columnwise
%
          [I J V] = find(tmpmat);
          nz = 0;               % compute the # of nonzero entries
          for k = 1:length(I)   % in the upper triangle of the block
             if J(k) >= I(k)
                nz = nz+1;
             end
          end
          fprintf(fp,'%d\n',nz);
          for k = 1:length(I)
             i = I(k);
             j = J(k);
             if j <= i    % save the lower triangle only
                fprintf(fp,'%d\n',i);           % row index
                fprintf(fp,'%d\n',j);           % column index
                fprintf(fp,'%20.16e\n',V(k));   % value
             end
          end    % k loop
       else    % matrix is full
          fprintf(fp,'%d\n',0);
          for i = 1:bsize
             for j = i:bsize
                fprintf(fp,'%20.16e\n',tmpmat(i,j));
             end
          end
       end   % if sparseblocks
    end    % blkidx loop: matrix C.s written to file
%
% write the matrices A1,...,Am
%
    for cstridx = 1:m
       Ak = smat(A.s(cstridx,:),blk.s);
       for blkidx = 1:nblk
          bsize = blk.s(blkidx);
          ulc = start(blkidx);
          lrc = ulc + bsize;
          ulc = ulc+1;
          tmpmat = Ak(ulc:lrc,ulc:lrc);
          if sparseblocks == 1
             fprintf(fp,'%d\n',1);
             [I J V] = find(tmpmat);
             nz = 0;               % compute the # of nonzero entries
             for k = 1:length(I)   % in the lower triangle of the block
                if J(k) >= I(k)
                   nz = nz+1;
                end
             end
             fprintf(fp,'%d\n',nz);
             for k = 1:length(I)
                i = I(k);
                j = J(k);
                if j <= i    % save the lower triangle only
                   fprintf(fp,'%d\n',i);    % row index
                   fprintf(fp,'%d\n',j);    % column index
                   fprintf(fp,'%20.16e\n',V(k));  % value
                end
             end    % k loop
          else    % blocks are full
             fprintf(fp,'%d\n',0);
             for i = 1:bsize
                for j = i:bsize
                   fprintf(fp,'%20.16e\n',tmpmat(i,j));
                end
             end
          end    % if sparseblocks
       end    % blkidx loop: matrix Ak written to file
    end    % cstridx loop
 end    % SD component
%
% next the QC data
%
 if isfield(blk,'q')
    nblk = length(blk.q);
    if sum(blk.q) <= 0
       nblk = 0;
    end
 else
    nblk = 0;
 end
 fprintf(fp,'%d\n',nblk);        % nblk = # blocks
 if nblk > 0                     % if nblk = 0, no QC component to problem
    fprintf(fp,'%d\n',blk.q);    % write block structure
%
% write the vector C.q and the matrix A.q
%
    fprintf(fp,'%20.16e\n',C.q);
    spblk = issparse(A.q);
    fprintf(fp,'%d\n',spblk);
    if spblk==1,
       [I J V] = find(A.q);
       fprintf(fp,'%d\n',length(I));
       fprintf(fp,'%d\n%d\n%20.16e\n',[I';J';V']);
    else
       fprintf(fp,'%20.16e\n',A.q);  % write A.q columnwise
    end
 end  % QC component
%
% and finally the LP data
%
 if isfield(blk,'l')
    nl = sum(blk.l);
 else
    nl = 0;
 end
 fprintf(fp,'%d\n',nl);        % size of LP block
%
% write the vectors C.l and the matrix A.l
%
 if nl > 0
    fprintf(fp,'%20.16e\n',C.l);
    spblk = issparse(A.l);
    fprintf(fp,'%d\n',spblk);
    if spblk==1,
       [I J V] = find(A.l);
       fprintf(fp,'%d\n',length(I));
       fprintf(fp,'%d\n%d\n%20.16e\n',[I';J';V']);
    else
       fprintf(fp,'%20.16e\n',A.l);  % write A.l columnwise
    end
 end  % LP component
%
 fp = fclose(fp);
 if fp == -1
    failed = 1;
 end
%
% END function
