function L = skrons(M,N,blk)
% SKRON  symmetric Kronecker product of M and N
%
% L = skrons(M,N,blk)
%
%   Suppose that blk = n nad let n2 = n*(n+1)/2 (the extension to
%   the multiple blocks case is straightfoward).
%   Then given 2 block diagonal matrices M, N, (not necessarily
%   symmetric) with block structure blk, skron computes their
%   symmetric Kronecker product, that is a square matrix L of
%   dimension n(n+1)/2 that represents the linear transformation
%       v |---> svec( (N smat(v) M^T + M smat(v) N^T)/2 ).
%   The columns of L are given by the image under L of the basis
%   of R^n2 corresponding to the canonical basis of the space of
%   symmetric matrices, namely the vectors w_ij, 1 <= i <= j <= n
%   given by
%                 [ Eii                 if i = j
%    smat(w_ij) = [
%                 [ (Eij + Eji)/sqrt(2) if i < j
%   where Eij is the matrix which is zero except for a 1 in the
%   (i,j)-entry.
%
%   Observe that:
%    [ N Eii M^T + M Eii N^T ](k,l) = N(k,i) M(l,i) + M(k,i) N(l,i)
%
%    [ N(Eij + Eji)M^T + M(Eij + Eji)N^T ](k,l) =
%      N(k,i)M(l,j) + N(k,j)M(l,i) + M(k,i)N(l,j) + M(k,j)N(l,i)
%
%    Let r2 = sqrt(2). Then indexing the columns of L with pairs
%    (i,j), 1 <= i <= j <= n, and the rows with pairs (k,l),
%    1 <= k <= l <= n, the ((k,l),(i,j))-entry of L is
%
%   if i = j and k = l:
%     (N(k,i)M(k,i)+(M(k,i)N(k,i))/2
%   if i = j and k < l:
%     [N(k,i)M(l,i)+M(k,i)N(l,i)]/r2
%   if i < j and k = l:
%     [N(k,i)M(k,j)+N(k,j)M(k,i)+M(k,i)N(k,j)+M(k,j)N(k,i))/(2*r2)
%   if i < j and k < l:
%     [N(k,i)M(l,j)+N(k,j)M(l,i)+M(k,i)N(l,j)+M(k,j)N(l,i)]/2
%
%   Note: for i <= j, the (i,j)-entry of a symmetric matrix M has
%   position
%     n(n+1)/2 - (n-i+1)(n-i+2)/2 + j - i + 1 = (i-1)(2n-i)/2 + j
%   in the vector svec(M).
%
%  input variables
%     - M         a block diagonal matrix (not necessarily symmetric)
%     - N         another block diagonal matrix (not necessarily symmetric)
%     - blk       block structure vector
%
%  output variables
%     - L         the symmetric Kronecker product of M and N

% SDPPACK Version 0.8 BETA
% Copyright (c) 1997 by
% F. Alizadeh, J.-P. Haeberly, M. Nayakkankuppam, M.L. Overton
% Last modified: 3/24/97
%
 nblk = length(blk);
 n2 = (sum(blk .* (blk+1)))/2;
 if nblk > 1,
    L = sparse(n2,n2);
 else,
    L = zeros(n2);
 end;
 r2 = sqrt(2);
%
 fin = 0;
 bfin = 0;
 for bidx = 1:nblk,
    start = fin + 1;
    bstart = bfin + 1;
    nb = blk(bidx);   % size of block
    nb2 = nb*(nb+1)/2;
    fin = fin + nb2;
    bfin = bfin + nb;
    Ltmp = zeros(nb2);
    Mb = M(bstart:bfin,bstart:bfin);
    Nb = N(bstart:bfin,bstart:bfin);
    for i = 1:nb,
       col = (i-1)*(2*nb - i)/2 + i;
       for k = 1:nb,
          row = (k-1)*(2*nb - k)/2 + k;
          Ltmp(row,col) = (Mb(k,i)*Nb(k,i) + Nb(k,i)*Mb(k,i))/2;
          for l = k+1:nb,
             row = row + 1;
             Ltmp(row,col) = (Mb(k,i)*Nb(l,i) + Nb(k,i)*Mb(l,i))/r2 ;
          end;
       end;
       for j = i+1:nb,
          col = col + 1;
          for k = 1:nb,
             row = (k-1)*(2*nb - k)/2 + k;
             Ltmp(row,col) = (Mb(k,i)*Nb(k,j) + Mb(k,j)*Nb(k,i) +...
                              Nb(k,i)*Mb(k,j) + Nb(k,j)*Mb(k,i))/(2*r2);
             for l = k+1:nb,
                row = row + 1;
                Ltmp(row,col) = (Mb(k,i)*Nb(l,j) + Mb(k,j)*Nb(l,i) +...
                                 Nb(k,i)*Mb(l,j) + Nb(k,j)*Mb(l,i))/2;
             end;
          end;
       end;
    end;
    L(start:fin,start:fin) = Ltmp;
 end;
%
% END function
