% SDP  driver for the function fsdp.m
%
%   Semidefinite Programming Solver using the XZ + ZX method
%
%   The following variables must be available in the Matlab workspace:
%      - A            matrix of primal constraints
%      - b            rhs of primal constraints
%      - C            cost matrix
%      - blk          block structure vector
%      - X            initial guess for primal variable
%      - y            initial guess for dual variable
%      - Z            initial guess for dual slack
%      - maxit        maximum iterations allowed
%      - tau          fraction of step to boundary of psd cone
%      - steptol      tolerance for short steps
%      - abstol       absolute tolerance on gap
%      - reltol       relative tolerance on gap
%      - gapprogtol   tolerance on sufficient progress
%      - feasprogtol  tolerance on sufficient progress
%      - bndtol       upper bound on size of solution
%      - prtlevel     verbosity level (accessed only by fsdp.m)
%

% SDPPACK Version 0.8 BETA
% Copyright (c) 1997 by
% F. Alizadeh, J.-P. Haeberly, M. Nayakkankuppam, M.L. Overton
% Last modified: 3/24/97
%
% termination flag:
%    termflag = -2  ==> X is blowing up
%    termflag = -1  ==> Z is blowing up
%    termflag =  0  ==> success
%    termflag =  1  ==> new point rejected since X indefinite: chol(X) failed
%    termflag =  2  ==> new point rejected since Z indefinite:
%                          chol(Z) failed or Z has eigenvalues <= 0
%    termflag =  3  ==> SchurComp numerically singular
%    termflag =  4  ==> new point rejected: worse than current point
%    termflag =  5  ==> shortsteps
%    termflag =  6  ==> exceeded maximum number of iterations
%    termflag =  7  ==> data failed validation test
%
% Also, if maxit > 0:
%    termflag =  1 and iter = 0 ==> initial X is not positive definite
%    termflag =  2 and iter = 0 ==> initial Z is not positive definite
%    termflag =  3 and iter = 0 ==> A was rank deficient; use preproc
%    termflag =  4 and iter = 0 ==> Initial pt. may be too close to bdry
%    termflag =  5 and iter = 0 ==> Initial pt. may be too close to bdry
%
% If maxit = 0, we assume the user wants to do just data validation.
%    termflag =  6  ==> data passed validation test
%
 flops(0);
 tstart = clock;
 cputstart = cputime;
%
 tau_orig = tau;
 scalefac_orig = scalefac;
 for runs = 1:3,
    fprintf('\ntau = %8.4f,     scalefac = %8.0f\n\n',tau,scalefac);
    [X,y,Z,iter,gapval,feasval,objval,termflag] = ...
        fsdp(A,b,C,blk,X,y,Z,maxit,tau,steptol,abstol,reltol,...
             gapprogtol,feasprogtol,bndtol,prtlevel,validate);
    if iter == 0,
       if termflag == 4 | termflag == 5,
          fprintf('\nsdp: Initial point may be too close to boundary\n');
       elseif termflag == 6 & maxit == 0,
          fprintf('\nsdp: Data passed validation tests\n');
       elseif termflag == 7 & prtlevel == 0,
          fprintf('\nsdp: Data failed validation tests\n');
          fprintf('For more information, rerun sdp with prtlevel = 1\n');
       end;
       break;
    end;

    if autorestart & (termflag == 4 | termflag == 5)...
      & (gapval(iter+1) + feasval(iter+1,1) + feasval(iter+1,2) > 1.0e-3),
       if runs < 3,
          fprintf('\nrestarting...\n');
       end;
       if runs == 1,
          tau = min(tau,.99);
       elseif runs == 2,
          tau = min(tau,.9);
       end;
       scalefac = 100*scalefac;
       initvars;
    else,
       break;
    end;
 end;   % runs
 tau = tau_orig;
 scalefac = scalefac_orig;
%
 if iter > 0,
    if termflag == 6,
       fprintf('\nsdp: reached maximum number of iterations\n');
    end;
    gapval = gapval(1:(iter+1));
    feasval = feasval(1:(iter+1),:);
    objval = objval(1:(iter+1),:);
    cputend = cputime - cputstart;
    elapsedtime = etime(clock,tstart);
    if runs > 1,
       fprintf('\nsdp: number of runs = %3.0f\n',min(runs,3));
    end;
    fprintf('\nsdp: elapsed time               = %- 9.3f seconds\n',elapsedtime);
    fprintf('sdp: elapsed cpu time           = %- 9.3f seconds\n',cputend);
    fprintf('sdp: flops                      = %- 13.0f\n',flops);
    fprintf('sdp: number of iterations       = %- 4.0f\n',iter);
    fprintf('sdp: final value of X.Z         = %- 11.3e\n',gapval(iter+1));
    fprintf('sdp: final primal infeasibility = %- 11.3e\n',feasval(iter+1,1));
    fprintf('sdp: final dual infeasibility   = %- 11.3e\n',feasval(iter+1,2));
    fprintf('sdp: primal objective value     = %- 24.16e\n',objval(iter+1,1));
    fprintf('sdp: dual objective value       = %- 24.16e\n',objval(iter+1,2));
 end;
