function [X,y,Z,iter,gapval,feasval,objval,termflag] = ...
   fsdp(A,b,C,blk,X,y,Z,maxit,tau,steptol,abstol,reltol,...
        gapprogtol,feasprogtol,bndtol,prtlevel,validate)
% FSDP   Semidefinite Programming Solver using the XZ + ZX method
%
% [X,y,Z,iter,gapval,feasval,objval,termflag] = ...
%    fsdp(A,b,C,blk,X,y,Z,maxit,tau,steptol,abstol,reltol,...
%       gapprogtol,feasprogtol,bndtol,prtlevel,validate)
%
%  input variables
%     - A            matrix of primal constraints
%     - b            rhs of primal constraints
%     - C            cost matrix
%     - blk          block structure vector
%     - X            initial guess for primal variable
%     - y            initial guess for dual variable
%     - Z            initial guess for dual slack
%     - maxit        maximum iterations allowed
%     - tau          fraction of step to boundary of psd cone
%     - steptol      tolerance for short steps
%     - abstol       absolute tolerance on gap
%     - reltol       relative tolerance on gap
%     - gapprogtol   tolerance on sufficient progress
%     - feasprogtol  tolerance on sufficient progress
%     - bndtol       upper bound on size of solution
%     - prtlevel     verbosity level
%     - validate     data validation flag
%
%  output variables
%     - X            computed primal solution
%     - y            computed dual solution
%     - Z            computed dual slack solution
%     - iter         the number of iterations performed by the algorithm
%     - gapval       vector of gap values of length iter+1
%     - feasval      matrix of primal and dual feasibility residuals, of size (iter+1)x2
%     - objval       matrix of primal and dual objective values, of size (iter+1)x2
%     - termflag     termination flag

% SDPPACK Version 0.8 BETA
% Copyright (c) 1997 by
% F. Alizadeh, J.-P. Haeberly, M. Nayakkankuppam, M.L. Overton
% Last modified: 3/24/97

% termination flag (iter > 0):
%    termflag = -2  ==> X is blowing up
%    termflag = -1  ==> Z is blowing up
%    termflag =  0  ==> success
%    termflag =  1  ==> new point rejected since X is indefinite: chol(X) failed
%    termflag =  2  ==> new point rejected since Z is indefinite:
%                          chol(Z) failed or Z has eigenvalues <= 0
%    termflag =  3  ==> SchurComp numerically singular
%    termflag =  4  ==> new point rejected: worse than current point
%    termflag =  5  ==> shortsteps
%    termflag =  6  ==> exceeded maximum number of iterations
%    termflag =  7  ==> data failed validation test
%
% Also:
%    termflag =  1 and iter = 0 ==> initial X is not positive definite
%    termflag =  2 and iter = 0 ==> initial Z is not positive definite
%    termflag =  3 and iter = 0 ==> A may be rank deficient; try preproc
%    termflag =  4 and iter = 0 ==> Initial pt. may be too close to the bdry
%    termflag =  5 and iter = 0 ==> Initial pt. may be too close to the bdry
%
% If maxit = 0, we assume the user wants to do just data validation.
%    termflag =  6  ==> data passed validation tests
%
 termflag = 7;
 iter = 0;
%
% check number of input arguments
%
 if nargin < 16 | nargin > 17,
    if prtlevel > 0,
       fprintf('fsdp: Incorrect number of input arguments.  Aborting...\n');
    end;
    return;
 end;
 if nargin == 16,
    validate = 0;
 end;
%
% some simple data validation
%
 m = length(b);
 nblk = length(blk);
 n = sum(blk);
 sumblk2 = sum(blk .* (1+blk))/2;
 if m > sumblk2,
    if prtlevel > 0,
       fprintf('fsdp: too many contraints. Try "help preproc"\n');
    end;
    return;
 end;
 if m ~= size(A,1),
    if prtlevel > 0,
       fprintf('fsdp: # contraints not equal to # rows of A\n');
    end;
    return;
 end;
 if m ~= length(y),
    if prtlevel > 0,
       fprintf('fsdp: y is improperly dimensioned\n');
    end;
    return;
 end;
 if size(A,2) ~= sumblk2
    if prtlevel > 0,
       fprintf('fsdp: # columns of A is incompatible with blk\n');
    end;
    return;
 end;
 if (n ~= length(C) | size(C,1) ~= size(C,2)),
    if prtlevel > 0,
       fprintf('fsdp: C is improperly dimensioned\n');
    end;
    return;
 end;
 if (n ~= length(X) | size(X,1) ~= size(X,2)),
    if prtlevel > 0,
       fprintf('fsdp: X is improperly dimensioned\n');
    end;
    return;
 end;
 if (n ~= length(Z) | size(Z,1) ~= size(Z,2)),
    if prtlevel > 0,
       fprintf('fsdp: Z is improperly dimensioned\n');
    end;
    return;
 end;
%
% now check that X and Z are block diagonal if validate flag is set
%
 if validate,
    if nblk > 1,
       if size(blk,1) ~= nblk,   % make sure blk is a column vector
          blk = blk';
       end;
       getblk = zeros(n,1);
       cblk(1) = blk(1);               % entries of cblk are the cumulative
       for i = 2:nblk,                 % sums of the block sizes
          cblk(i) = cblk(i-1) + blk(i);
       end;
       for i = 1:n               % for a given row/column index i, find
          t = find(i <= cblk);   % the index of the block it belongs to
          getblk(i) = t(1);
       end;
       [I,J] = find(X);
       idx1 = getblk(I);
       idx2 = getblk(J);
       loc = max(idx1 ~= idx2);
       if loc > 0,
          if prtlevel > 0,
             fprintf('fsdp: X has invalid block structure\n');
          end;
          return;
       end;
       [I,J] = find(Z);
       idx1 = getblk(I);
       idx2 = getblk(J);
       loc = max(idx1 ~= idx2);
       if loc > 0,
          if prtlevel > 0,
             fprintf('fsdp: Z has invalid block structure\n');
          end;
          return;
       end;
    end;
 end;   % end of all validation
%
% Compute chol(X), eigenvectors and eigenvalues of Z and Evsum so they
% are available on entry to the loop
%
 [chol_X,indef] = chol(X);
 if indef > 0,
    if prtlevel > 0,
       fprintf('fsdp: initial X is not positive definite. Aborting...\n');
    end;
    termflag = 1;
    return;
 end;
%
 [chol_Z,indef] = chol(Z);
 if indef > 0,
    if prtlevel > 0,
       fprintf('fsdp: initial Z is not positive definite. Aborting...\n');
    end;
    termflag = 2;
    return;
 end;
 [Zeigval,Zeigvec,indef] = blkeig(Z,blk);  % note: Zeigval is a vector
 if indef == 1,            % if Znew is indefinite, we quit
    termflag = 2;          % and return the previous iterate
    if prtlevel > 0,
       fprintf('fsdp: initial Z has nonpositive eigenvalues. Aborting...\n');
    end;
    return;
 end;
%
% Compute Evsum from the eigenvalues of Z:
%        Evsum(i,j) = 1/(Zeigval(i) + Zeigval(j));
% Evsum is needed to solve the Lyapunov system
%
 if nblk > 1,
    Evsum = sparse(n,n);
 else,
    Evsum = zeros(n,n);
 end;
 start = 1;  fin = 0;
 for i=1:nblk
    bsize = blk(i);
    fin = fin + bsize;
    tmp = Zeigval(start:fin);
    tmp = tmp(:,ones(1,bsize));
    Evsum(start:fin,start:fin) = tmp + tmp';
    start = start + bsize;
 end;
%
% Now some initialization
%
 vC = svec(C,blk);
 vX = svec(X,blk);
 vZ = svec(Z,blk);
 B = zeros(size(A'));     % kth column will be solution of a Lyapunov eqn
 Id2 = 2*speye(n);
 dgap = full(vX'*vZ);
 rp = b - A*vX;
 pri_infeas = norm(rp);
 Rd = vC - vZ - A'*y;     % a vector
 dual_infeas = norm(Rd);
 Rd = smat(Rd,blk);
 gapval(1) = dgap;
 objval(1,1) = full(vC'*vX);
 objval(1,2) = b'*y;
 feasval(1,1) = pri_infeas;
 feasval(1,2) = dual_infeas;
 Xstep = 0;
 Zstep = 0;
 termflag = 6;
%
% print header and initial info
%
 if prtlevel > 0,
    fprintf('iter   p_step      d_step     p_infeas    d_infeas      X.Z ');
    fprintf('        pobj        dobj\n');
    fprintf('%3.0f %11.3e %11.3e', 0, 0.0, 0.0);
    fprintf(' %11.3e %11.3e %11.3e', pri_infeas, dual_infeas, dgap);
    fprintf(' %11.3e %11.3e\n',objval(1,1),objval(1,2));
 end;
%
% start main loop
%
 for iter = 1:maxit;
%
% construct Schur complement
%
    for k=1:m,
       Ak = smat(A(k,:),blk);  % the matrix Ak is the kth row of A
       rhsk = X*Ak;
       rhsk = rhsk + rhsk';
% and now solve ZY + YZ = rhsk
       rhsk = Zeigvec'*rhsk*Zeigvec;
       lyapsol = Zeigvec*evsumdiv(rhsk,Evsum,blk)*Zeigvec';
       B(:,k) = svec(lyapsol,blk);
    end;
    SchurComp = A*B;   % NOT symmetric
%
% compute its LU factorization
%
    [Lfactor,Ufactor,Perm] = lu(SchurComp);
%
% check if SchurComp is numerically singular
%
    minpiv = min(abs(diag(Ufactor)));
    if minpiv == 0,
       if prtlevel > 0,
          fprintf('fsdp: stop since limiting accuracy reached.\n');
          fprintf('      (Schur complement is numerically singular)\n');
          fprintf('      (A may be rank deficient.  Try running preproc)\n');
       end;
       termflag = 3;
       iter = iter-1;
       return;
    end;
%
% predictor step
%
    XZ = X*Z;
% predictor step does not use mu
    Rc = -(XZ + XZ');
%
    [dX,dy,dZ] =...
       sdpsol(X,A,Zeigvec,Evsum,Rd,rp,Rc,Lfactor,Ufactor,Perm,blk);
%
% symmetrize dX and dZ : good idea because of rounding
%
    dX = 0.5*(dX+dX');
    dZ = 0.5*(dZ+dZ');
%
% compute new gap
%
    Xstep = min(1,tau*sdbound(chol_X,dX,blk));  % tau times
    Zstep = min(1,tau*sdbound(chol_Z,dZ,blk));  % step to boundary
    Xnew = X + Xstep*dX;
    Znew = Z + Zstep*dZ;
    gapnew = full(sum(sum(Xnew .* Znew)));
%
% corrector step
%
    mu = (gapnew/dgap)^2 * gapnew/n;     % Mehrotra's formula
    dXdZ = dX*dZ;
    Rc = Rc + mu*Id2 - (dXdZ + dXdZ');  % included mu term this time
%
% Compute the corrector step
%
    [dX,dy,dZ] =...
       sdpsol(X,A,Zeigvec,Evsum,Rd,rp,Rc,Lfactor,Ufactor,Perm,blk);
%
% symmetrize dX and dZ
%
    dX = 0.5*(dX+dX');
    dZ = 0.5*(dZ+dZ');
%
% and compute the stepsizes
%
    Xstep = min(1,tau*sdbound(chol_X,dX,blk));
    Zstep = min(1,tau*sdbound(chol_Z,dZ,blk));
%
% end of corrector step and iteration;
%
% decide whether to accept new point or quit
% failure if steps get too short
%
    if Xstep < steptol | Zstep < steptol,
       if prtlevel > 0,
          fprintf('fsdp: stop since steps are too short\n');
       end;
       termflag = 5;
       iter = iter-1;
       return;
    end;
%
    Xnew = X + Xstep*dX;
    Znew = Z + Zstep*dZ;
    ynew = y + Zstep*dy;
%
% check for positive definiteness of Xnew
%
    [chol_X,indef] = chol(Xnew);  % if Xnew is indefinite, we quit
    if indef > 0,                 % and return the previous iterate
       termflag = 1;
       if prtlevel > 0,
          fprintf('fsdp: stop since limiting accuracy reached.\n');
          fprintf('      (new X is indefinite, hence rejected)\n');
       end;
       iter = iter-1;
       return;
    end;
%
% check for positive definiteness of Znew
%
    [chol_Z,indef] = chol(Znew);  % if Xnew is indefinite, we quit
    if indef > 0,                 % and return the previous iterate
       termflag = 2;
       if prtlevel > 0,
          fprintf('fsdp: stop since limiting accuracy reached.\n');
          fprintf('      (new Z is indefinite, hence rejected)\n');
       end;
       iter = iter-1;
       return;
    end;
%
% find evecs and evals of Znew and Evsum (needed to solve
% the Lyapunov system) and test for nonpositive eigenvalues
%
    [Zeigval,Zeigvec,indef] = blkeig(Znew,blk);
    if indef == 1,            % if Znew is indefinite, we quit
       termflag = 2;          % and return the current iterate
    end;
    if termflag ~= 2,   % otherwise we are going to quit, so we don't need Evsum
       start = 1;  fin = 0;
       for i=1:nblk
          bsize = blk(i);
          fin = fin + bsize;
          tmp = Zeigval(start:fin);
          tmp = tmp(:,ones(1,bsize));
          Evsum(start:fin,start:fin) = tmp + tmp';
          start = start + bsize;
       end;
    end;
%
% decide whether new point provides "substantial"
% improvement over previous iterate
%
    vXnew = svec(Xnew,blk);
    vZnew = svec(Znew,blk);
    gapnew = full(vXnew'*vZnew);
    rpnew = b - A*vXnew;
    Rdnew = vC - vZnew - A'*ynew;
    pri_infeas_new = norm(rpnew);
    dual_infeas_new = norm(Rdnew);
    Rdnew = smat(Rdnew,blk);
%
% if gap is small, and feasibility is getting worse, then
% reject new point unless gap is greatly improved
%
    if gapnew < abstol & gapprogtol*gapnew > dgap & ...
       (pri_infeas_new > feasprogtol*pri_infeas | ...
        dual_infeas_new > feasprogtol*dual_infeas),
%
% reject the new iterate
%
       termflag = 4;
       if prtlevel > 0,
          fprintf('Stop since new point is substantially worse than current iterate\n');
          fprintf('      X.Z = %11.3e\n',gapnew);
          fprintf('      pri_infeas = %11.3e\n', pri_infeas_new);
          fprintf('      dual_infeas = %11.3e\n', dual_infeas_new);
       end;
       iter = iter-1;
       return;
    else,
%
% accept new point
%
       X =Xnew;
       y = ynew;
       Z = Znew;
       vX = vXnew;
       vZ = vZnew;
       dgap = gapnew;
       rp = rpnew;
       Rd = Rdnew;
       pri_infeas = pri_infeas_new;
       dual_infeas = dual_infeas_new;
%
% compute objective values so they can be plotted later
%
       gapval(iter+1) = dgap;
       objval(iter+1,1) = full(vC'*vX);
       objval(iter+1,2) = b'*y;
       feasval(iter+1,1) = pri_infeas;
       feasval(iter+1,2) = dual_infeas;
%
% and display information
%
       if prtlevel > 0,
          fprintf('%3.0f %11.3e %11.3e',iter,Xstep,Zstep);
          fprintf(' %11.3e %11.3e %11.3e',pri_infeas,dual_infeas,dgap);
          fprintf(' %11.3e %11.3e\n',objval(iter+1,1),objval(iter+1,2));
       end;
    end;
%
% termination test
%
    if termflag == 2,  % Z has eigs <= 0, so return current iterate
       if prtlevel > 0,
          fprintf('fsdp: stop since limiting accuracy reached');
          fprintf(' (smallest eigenvalue of Z = %11.3e)\n', min(Zeigval));
       end;
       return;
    end;
    total_err = dgap + pri_infeas + dual_infeas;
    normX = norm(vX);
    normZ = norm(vZ);
    if total_err < min(abstol,reltol*(normX + normZ)),
       if prtlevel > 0,
          fprintf('fsdp: stop since error reduced to desired value\n');
       end;
       termflag = 0;
       return;
    elseif normX > bndtol,
       if prtlevel > 0,
          fprintf('fsdp: stop since norm of X exceeds bndtol\n');
       end;
       termflag = -2;
       return;
    elseif normZ > bndtol,
       if prtlevel > 0,
          fprintf('fsdp: stop since norm of Z exceeds bndtol\n');
       end;
       termflag = -1;
       return;
    end;
 end;
%
% END function
