function [lam,W] = toepeig(c,tol)

% Approximate eig(T) with T = T' = toeplitz(c) of size (n,n) using asymptotics
% by Kac, Murdock, Szegoe, Widom and Parter.
% Notation is consistent with [1 sec. 5], [2 sec. 4] and [3 sl. 108 ff].
%
% Needs O(N*log(N)) time and O(N) space, N=2*n+2.
%
% [1] Albrecht Boettcher, Egor A. Maksimenko and Sergei M. Grudsky,
% On the structure of the eigenvectors of large Hermitian Toeplitz band matrices
% Operator Theory: Advances and Applications 210 (2010), 15-36.
% [2] Albrecht Boettcher, Egor A. Maksimenko and Sergei M. Grudsky,
% Inside the eigenvalues of certain Hermitian Toeplitz band matrices
% Journal of Computational and Applied Mathematics 233 (2010), 2245-2264.
% [3] Sergei M. Grudsky, Eigenvalues of larger Toeplitz Matrices, talk, 2010.
%
% Copyright (c) by Hannes Nickisch and Andrew Wilson, 2015-01-08.

if nargin<2, tol = 1e-4; end                                 % set default value
c = c(:); n = size(c,1);
r = n-find(abs(flipud(c))/max(abs(c))>tol,1,'first'); if isempty(r), r=n-1; end
m = 1:n; j = m'; x = pi*j/(n+1); p = x;                     % half circle (0,pi)
c = c(1:r+1);                                        % retain only relevant part

% fast FFT based init using equispaced x, exact for tridiagonal T
lam = fft(c,2*n+2); lam = 2*real(lam(2:n+1))-c(1);    % same as lam = psi(x,c,r)
imax = 5; i = 0;
% xn = linspace(0,pi,501)'; yn = theta(psi(xn,c,r),c,r); thspl = spline(xn,yn);
while i<imax                  % iterate until convergence, [2 sec. 4, 3 sl. 120]
  if r<2, break, end
%   th = ppval(thspl,lam);
  [th,U] = theta(lam,c,r);
  p = x-th/(n+1); lam = psi(p,c,r);                               % p = phi(lam)
  i = i+1;
end
lam = max(lam,0);                              % allow only positive eigenvalues
[th,U] = theta(lam,c,r); p = x-th/(n+1); lam = psi(p,c,r);

o = ones(n,1); W = sin((j-(n+1)/2)*p'+o*(m*pi/2));
if i==0                                    % exact eigenvector for tridiagonal T
  W = sqrt(2/(n+1)) * W;                             % closed-form normalisation
else                                             % eigenvectors use [3 sl. 134].
  eip = exp(1i*p'); [H,dH] = h(eip,U);
  Q = (U-repmat(eip,r-1,1)).*(U-repmat(1./eip,r-1,1)).*dH;
  Q = repmat(abs(H).*sin(p'),r-1,1)./ Q;
  J1 = repmat(-j,1,r-1); Jn = repmat(j-n-1,1,r-1);                     % indices
  for l=1:n                        % subtract Q term, one vector after the other
    u = o*U(:,l).'; W(:,l) = W(:,l) - (u.^J1-(-1)^l*u.^Jn) * Q(:,l);
  end
  W = W./repmat(sqrt(sum(W.*W,1)),n,1);                              % normalise
end

function [th,U] = theta(lam,c,r)                                      % O(r^3*n)
  n = numel(lam); th = zeros(n,1); c2 = c(1+(1:r));
  if nargout>1, U = zeros(r-1,n); end
  for i=1:numel(lam)
    a = [flipud(c2); c(1)-lam(i); c2]; % coefficients of real Laurent polynomial
    z = sort(roots(a),'descend');                       % compute and sort roots
    zr = z(r); if imag(zr)<0, zr = 1/zr; end
    th(i) = 2*angle(h(zr,z(1:r-1)));
    if nargout>1, U(:,i) = z(1:r-1); end
  end

function [y,dyU] = h(z,U)                             % y = h(z), dyU = dh(U)/dz
  o = ones(size(U,1),1);
  y = prod(1-(o*z)./U,1);
  if nargout>1
    r = size(U,1)+1; dyU = zeros(size(U));
    for i=1:r-1
      ni = [1:i-1,i+1:r-1]; ui = U(i,:);
      dyU(i,:) = -prod(1-repmat(ui,r-2,1)./U(ni,:),1)./ui;
    end
  end

% Symbol of T: a(t) = c(1) + sum[j=2:n] c(j)*( t^(1-j) + t^(j-1) ).
% g(x) = a( exp(1i*x) ) = c(1) + 2*sum[j=2:n] c(j)*cos((j-1)*x).
% psi: [0,pi] -> [0,M] is even and monotonic with psi(0)=0 and psi(pi)=M
function lam = psi(x,c,r)        % lam=psi(x)=g(x), x=phi(lam)=g^-1(lam), O(r*n)
  lam = c(1)*ones(size(x)) + 2*cos(x*(1:r))*c(2:r+1);