/* Solve y = T*x of y of size (n,k) with T = toeplitz(c,r) of size (n,n) using
 * Levinson recursion.
 * Needs O(4*n^2*k) time and O(n*k) space.
 *
 * Copyright (c) by Hannes Nickisch, 2014-12-09.                              */
#include <math.h>
#include <mex.h>

#define	 C_IN	  prhs[0]
#define	 R_IN	  prhs[1]
#define	 Y_IN	  prhs[2]
#define	 X_OUT	plhs[0]

void mexFunction( int nlhs, mxArray *plhs[], int nrhs, const mxArray*prhs[] )
{
  double *c, *r, *x, *y, *f, *b, ex, ef, eb, w, b0, f0, bj;        /* declare */
	int i,j,l,n,k;

	if (nrhs != 3) mexErrMsgTxt("Three input arguments required.");    /* check */
  else if (nlhs > 1) mexErrMsgTxt("Only one output argument available."); 

  n = mxGetNumberOfElements(C_IN);    c = mxGetPr(C_IN);            /* assign */
  r = mxGetPr(R_IN);
  if (mxGetNumberOfElements(R_IN)==0) r = mxGetPr(C_IN);
  else if (mxGetNumberOfElements(R_IN)!=n)
    mexErrMsgTxt("r needs to be of the same size as c.");
  if (mxGetM(Y_IN)!=n) mexErrMsgTxt("x needs to match the size of c and r.");
  y = mxGetPr(Y_IN); k = mxGetN(Y_IN);
  X_OUT = mxCreateDoubleMatrix(n,k,mxREAL); x = mxGetPr(X_OUT);

  f = (double*) mxMalloc(n*sizeof(double));                       /* allocate */
  b = (double*) mxMalloc(n*sizeof(double));
  
  for (l=0; l<k; l++) {                                 /* iterate over k rhs */
    f[0] = 1/c[0]; b[0] = f[0]; x[0] = y[0]*f[0];                     /* init */
    for (i=1; i<n; i++) {                                          /* compute */
      ex = 0.0; ef = 0.0; eb = 0.0;
      for (j=0; j<i; j++) {
        ex += c[i-j]*x[j];
        ef += c[i-j]*f[j];
        eb += r[1+j]*b[j];
      }
      w = 1-eb*ef;
      for (j=0; j<=i; j++) {
        if (j==0) b0 = 0; else b0 = bj;
        if (j==i) f0 = 0; else f0 = f[j];
        f[j] = (f0-ef*b0)/w;
        bj = b[j];
        b[j] = (b0-eb*f0)/w;
      }
      for (j=0; j<=i; j++) x[j] += (y[i]-ex)*b[j];
    }
    x += n; y += n;
  }

  mxFree(f); mxFree(b);                                            /* tidy up */
}