/* Compute s of size (n,1) with s(i) = sum(diag(inv(T),i-1))
 * where T = toeplitz(r) of size (n,n) using Trench's algorithm.
 * The input v is obtained by the call [logdetT,v] = toeplogdet(r).
 * Zhang et al., Time-series GPR Based on Toeplitz Computation, 2005, alg 3.1
 * Needs O(13*n^2/4) time and O(n) space.
 *
 * Copyright (c) by Hannes Nickisch, 2014-12-09.                              */
#include <math.h>
#include <mex.h>

#define	 V_IN	  prhs[0]
#define	 S_OUT	plhs[0]

void mexFunction( int nlhs, mxArray *plhs[], int nrhs, const mxArray*prhs[] )
{
  double *v, *s, w;                                                /* declare */
	int i,j,k,n;

	if      (nrhs != 1) mexErrMsgTxt("One input argument required.");  /* check */
  else if (nlhs >  1) mexErrMsgTxt("At most one output argument available."); 

  n = mxGetNumberOfElements(V_IN);  v  = mxGetPr(V_IN);             /* assign */
  S_OUT = mxCreateDoubleMatrix(n,1,mxREAL); s = mxGetPr(S_OUT);

  for (i=0; i<n; i++) s[i] = 0.0;                                     /* init */
  
  for (k=0; k<n; k++) {                             /* iterate over diagonals */
    for (j=0; j<floor(0.5*(n-1-k))+1; j++) {
      i = j+k;
      if (j==0) w = v[n-1-i];
      else      w += (v[n-1-j]*v[n-1-i]-v[i-1]*v[j-1])/v[n-1];
      if (i==n-1-j) s[k] +=   w;
      else          s[k] += 2*w;
    }
  }
}