/* This file is part of the source code for the following publication:
 * Assembling Self-Supporting Structures, Deuss et al., SIGGRAPH Asia 2014
 *
 * Copyright (C) 2014 Mario Deuss <mario.deuss@epfl.ch>
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */

#ifndef SELFASSEMBLYDEFINES_H
#define SELFASSEMBLYDEFINES_H

#include <Eigen/Dense>
#include <Eigen/SparseCore>
#include <vector>
#include <limits>

namespace SelfAssembly{

#define MYOUT std::cout
#define MYERR std::cerr
#define MYNAN std::numeric_limits<double>::quiet_NaN()
#define MYINF std::numeric_limits<double>::infinity()

inline double numericalZero(){ return 10e-8;}

//Indices
typedef int Idx;
typedef std::vector<Idx> Idxs;
inline Idx invalidIdx(){return -1;}

//Vertex/Vertices
typedef Eigen::Vector3d V;
typedef std::vector<V> Vs;
inline V vNan(){ return V::Constant(MYNAN); }
inline V gravity(){ return V(0.,0.,-9.81);}

//Vectors:
typedef double Scalar;
typedef Eigen::Vector2d Vector2;
typedef Eigen::Vector3d Vector3;
typedef Eigen::Vector4d Vector4;

//Triangle(s) Vertex Indices
typedef Eigen::Matrix<Idx,3,1> T;
typedef std::vector<T> Ts;

//Force vector(s)
typedef Eigen::Vector3d Force;
typedef std::vector<Force> Forces;

//Triplets (for sparse matrices)
typedef Eigen::Triplet<double> Triplet;
typedef std::vector<Triplet> Triplets;

//sum_{i,j}|m_ij|^p
inline double lpNormPowP(const Eigen::Ref<Eigen::MatrixXd>& m, double p){
	double lpn=0.0;
	for(unsigned int i=0;i<m.rows();++i){
		for(unsigned int j=0;j<m.cols();++j){
			lpn += std::pow(std::fabs(m(i,j)),p);
		}
	}
	return lpn;
}

}

#endif // SELFASSEMBLYDEFINES_H
