% This is a simple example of loading a triangle mesh model and constrain
% it between two planes.
%
% Author: Christian Schller
% Email: schuellc[@]inf.ethz.ch
% Project page: http://igl.ethz.ch/projects/ams/
% 
% Hint: Z-ordering constraints are currently not implemented, therefore the result
% can have self-intersections. You could add them by finding all ray/triangle
% intersections, where the rays are shoot through all the vertices of the model
% and formulate them as linear inequality constraints in the matrix zOrder.
addpath(genpath('./toolbox/'));
close all;

% input triangle mesh model
[V,T] = readOBJ('bunnyLow.obj');
n = size(V,1);

% initial model front view
figure;
subplot(2,2,1);
showMesh(V,T,0,90);
subplot(2,2,2);
showMesh(V,T,-30,10);

% viewpoint [x,y,z]
vp = [0,0,-10];

% normal weighting vector (n x 1) [weightFactor]
% n : number of vertices
% weightFactor : scalar value
w = ones(n,1);

% thickness constraints vector (c x 2) [vertexIdx,lambdaValue]
% c           : number of constraints
% vertexIdx   : index of the vertex in V
% lambdaValue : distance from the viewpoint to the coresponding vertex

maxDistPlaneZ = 0.1;
minDistPlaneZ = 0;

% Here we compute the lambdas for two limiting thickness constraint planes
V = V - repmat(vp,n,1);
R = normr(V);
lambdaMin = ((minDistPlaneZ-vp(3))*ones(n,1)) ./ R(:,3);
lambdaMax = ((maxDistPlaneZ-vp(3))*ones(n,1)) ./ R(:,3);
lambdaConMin = [[1:n]',lambdaMin];
lambdaConMax = [[1:n]',lambdaMax];

% Fixed vertices vector (f x 1) [vertexIdx]
% f : numbers of fixed vertices
% vertexIdx : index of the vertex in V
% (one for each disconnected component of the mesh)
vFixed = [1];

% mu assignment vector (n x 1) [muIdx]
% muIdx : index of the additional mu variables
% (one for each independent thickness constraint region)
mu = ones(n,1);

% z-order constraint matrix coefficients (z x 8) [order,vIdx,vAIdx,vBIdx,vCIdx,u,v,w]
% z : numbers of z-order constraints (vertex <-> point on triangle (barycentric coordinates) order)
% order : 1 for vertex in front of triangle; -1 for vertex behind triangle
% vIdx : index of vertex in V
% vAIdx : first vertex (A) index of the triangle
% vBIdx : second vertex (B) index of the triangle
% vCIdx : third vertex (C) index of the triangle
% u : barycentric coordinate related to vertex A
% v : barycentric coordinate related to vertex B
% w : barycentric coordinate related to vertex C
zOrder = []; % z-Order constraints are disabled

% create AMS
[V1,T] = createAMS(V,T,vp,lambdaConMin,lambdaConMax,vFixed,w,mu,zOrder);

% deformed model top view
subplot(2,2,3);
showMesh(V1,T,0,90);
subplot(2,2,4);
showMesh(V1,T,-30,10);

% save AMS model
% writeOBJ('models/result.obj',V1,T);