#include "SeamlessSimilarityMapsPlugin.hh"

#include <OpenFlipper/BasePlugin/PluginFunctions.hh>

#include "ConformalSeamlessSimilarityMapping.hh"

#include <algorithm>

//-----------------------------------------------------------------------------

void
SeamlessSimilarityMapsPlugin::
initializePlugin()
{
   tool_ = new SeamlessSimilarityMapsToolbarWidget();
   QSize size(300, 300);
   tool_->resize(size);

   connect(tool_->compute_pb, SIGNAL(clicked()), this, SLOT(slotCompute()));

   emit addToolbox( tr("SeamlessSimilarityMaps") , tool_ );
}


//-----------------------------------------------------------------------------

void
SeamlessSimilarityMapsPlugin::
slotCompute()
{
  for ( PluginFunctions::ObjectIterator o_it(PluginFunctions::TARGET_OBJECTS );
        o_it != PluginFunctions::objectsEnd(); ++o_it)
  {
    if ( o_it->dataType( DATA_TRIANGLE_MESH ) )
    {
      TriMesh* mesh = PluginFunctions::triMesh(*o_it);

      slotCompute( mesh, o_it->path() + "/" + o_it->filename() );
      
      o_it->hide();
    }
  }
  
  PluginFunctions::setDrawMode( ACG::SceneGraph::DrawModes::SOLID_2DTEXTURED_FACE_SHADED );
  
  emit updateView();
}

void
SeamlessSimilarityMapsPlugin::
slotCompute( TriMesh* _mesh, QString filename) {
  if ( _mesh == 0 ) return;
  
  Mesh m;
  
  std::cout << "Setup mesh data structure" << std::endl;
  
  m.out.resize(_mesh->n_vertices());
  m.n.resize(_mesh->n_halfedges());
  m.to.resize(_mesh->n_halfedges());
  m.f.resize(_mesh->n_halfedges());
  m.h.resize(_mesh->n_faces());
  m.l.resize(_mesh->n_edges());
  
  for(TriMesh::VertexIter vh = _mesh->vertices_begin(); vh != _mesh->vertices_end(); vh++)
    m.out[vh->idx()] = _mesh->halfedge_handle(*vh).idx();
  
  for(TriMesh::HalfedgeIter hh = _mesh->halfedges_begin(); hh != _mesh->halfedges_end(); hh++)
    m.n[hh->idx()] = _mesh->next_halfedge_handle(*hh).idx();
  
  for(TriMesh::HalfedgeIter hh = _mesh->halfedges_begin(); hh != _mesh->halfedges_end(); hh++)
    m.to[hh->idx()] = _mesh->to_vertex_handle(*hh).idx();
  
  for(TriMesh::HalfedgeIter hh = _mesh->halfedges_begin(); hh != _mesh->halfedges_end(); hh++)
  {
    m.f[hh->idx()] = _mesh->face_handle(*hh).idx();
    if(m.f[hh->idx()] < 0) { std::cerr << "ERROR: mesh has boundary; not supported yet." << std::endl; return; }
  }
  
  for(TriMesh::FaceIter fh = _mesh->faces_begin(); fh != _mesh->faces_end(); fh++)
    m.h[fh->idx()] = _mesh->halfedge_handle(*fh).idx();
  
  for(TriMesh::EdgeIter eh = _mesh->edges_begin(); eh != _mesh->edges_end(); eh++)
    m.l[eh->idx()] = _mesh->calc_edge_length(*eh);
  
  m.init();
  
  std::vector<double> Theta_hat(m.n_vertices(), 2.0*M_PI);
  std::vector<double> kappa_hat;
  std::vector< std::vector<int> > gamma;
  
  int genus = 1 - (_mesh->n_vertices() - _mesh->n_edges() + _mesh->n_faces())/2;
  std::cerr << "Genus = " << genus << std::endl;
  
  std::ifstream fin(filename.append(QString(".sig")).toStdString().c_str());
  if(fin)
  {
    std::cout << "Loading signature from .sig file:" << std::endl;
    
    std::string line;
    getline(fin, line);
    
    int i;
    double d;
    std::istringstream is(line);
    while(is >> i)
    {
      if(!(is >> d)) { std::cerr << "ERROR A\n"; return; }
      Theta_hat[i] = d;
      std::cerr << "  Theta_hat["<<i<<"] = "<<d<<std::endl;
    }
    kappa_hat.resize(2*genus);
    gamma.resize(2*genus);
    for(int s = 0; s < 2*genus; s++)
    {
      getline(fin, line);
      std::istringstream is(line);
      while(is >> i) gamma[s].push_back(i);
      getline(fin, line);
      std::istringstream is2(line);
      if(!(is2 >> d)) { std::cerr << "ERROR B\n"; return; }
      kappa_hat[s] = d;
      std::cerr << "  kappa_hat["<<s<<"] = "<<d<<std::endl;
    }
    fin.close();
  }
  else
  {
    std::cout << "Distributing some simple cones randomly" << std::endl;
    
    int n_v = _mesh->n_vertices();
    std::srand(0);
    std::vector<int> permutation(n_v);
    for(int i = 0; i < n_v; i++)
      permutation[i] = i;
    std::random_shuffle(permutation.begin(), permutation.end());
    if(genus == 0)
    {
      for(int i = 0; i < 8; i++)
        Theta_hat[permutation[i]] = M_PI/2*3; // "valence 3" cone
    }
    else
    {
      for(int i = 0; i < (genus-1)*8; i++)
        Theta_hat[permutation[i]] = M_PI/2*5; // "valence 5" cone
    }
    
    std::cout << "Computing non-contractible cycle basis" << std::endl;
    
    gamma = noncontractible_cycle_basis(*_mesh);
    kappa_hat.resize(gamma.size(), 0.0);
    int n_s = gamma.size();
    for(int s = 0; s < n_s; s++) // set kappa_hat to closest multiple of PI/2
    {
      int loop_size = gamma[s].size();
      for(int si = 0; si < loop_size; si++)
      {
        int h = gamma[s][si];
        int hn = m.n[h];
        int hnn = m.n[hn];
        if(m.opp(hn) == gamma[s][(si+1)%loop_size])
          kappa_hat[s] -= _mesh->calc_sector_angle(OpenMesh::HalfedgeHandle(h));
        else if(m.opp(hnn) == gamma[s][(si+1)%loop_size])
          kappa_hat[s] += _mesh->calc_sector_angle(OpenMesh::HalfedgeHandle(hnn));
        else { std::cerr << "ERROR: loop is broken." << std::endl; return; }
      }
      kappa_hat[s] = std::round(kappa_hat[s]/(M_PI/2))*(M_PI/2);
    }
    if((int)gamma.size() != 2*genus) { std::cerr << "ERROR: incorrect number of loops." << std::endl; return; }
    
  }
  
  std::cout << "Computing conformal seamless similarity parametrization" << std::endl;
  
  ConformalSeamlessSimilarityMapping mapping(m, Theta_hat, kappa_hat, gamma);
  std::vector<double> u, v;
  mapping.compute(u, v);
  
  std::cout << "Getting (possibly modified) output mesh" << std::endl;
  
  std::vector<int> n;
  std::vector<int> to;
  std::vector<int> f;
  std::vector<int> h;
  std::vector<int> out;
  m.get_mesh(n, to, f, h, out);
  
  BaseObjectData* newMesh;
  int newID;
  emit addEmptyObject(DATA_POLY_MESH, newID);
  if(!PluginFunctions::getObject(newID, newMesh)) { std::cerr << "Couldn't get new mesh!\n"; return; }
  newMesh->setName(QString("Output Mesh"));
  newMesh->target(true);
  PolyMesh& newmesh = *PluginFunctions::polyMesh(newMesh);
  
  for(unsigned int i = 0; i < out.size(); i++)
    newmesh.set_point(newmesh.new_vertex(), _mesh->point(OpenMesh::VertexHandle(i)));
  for(unsigned int i = 0; i < n.size(); i += 2)
  {
    int tov = to[i];
    int fromv = to[i+1];
    newmesh.new_edge(OpenMesh::VertexHandle(fromv), OpenMesh::VertexHandle(tov));
  }
  for(unsigned int i = 0; i < h.size(); i++)
    newmesh.new_face();
  for(unsigned int i = 0; i < n.size(); i++)
  {
    OpenMesh::HalfedgeHandle hi(i);
    newmesh.set_next_halfedge_handle(hi, OpenMesh::HalfedgeHandle(n[i]));
    newmesh.set_face_handle(hi, OpenMesh::FaceHandle(f[i]));
    newmesh.set_vertex_handle(hi, OpenMesh::VertexHandle(to[i]));
  }
  for(unsigned int i = 0; i < out.size(); i++)
    newmesh.set_halfedge_handle(OpenMesh::VertexHandle(i), OpenMesh::HalfedgeHandle(out[i]));
  for(unsigned int i = 0; i < h.size(); i++)
    newmesh.set_halfedge_handle(OpenMesh::FaceHandle(i), OpenMesh::HalfedgeHandle(h[i]));
  
  newmesh.update_normals();
  
  std::vector<double> ui, vi;
  ui = m.interpolate(u);
  vi = m.interpolate(v);
  
  std::cout << "Visualize map using texture" << std::endl;
  
  newmesh.request_halfedge_texcoords2D();
  emit addTexture( "SeamlessSimilarityMap", "quadTexture.png", 2, newID);
  emit setTextureMode( "SeamlessSimilarityMap", "type=halfedgebased", newID);
  emit switchTexture( "SeamlessSimilarityMap", newID);
  
  int n_u = ui.size();
  OpenMesh::Vec2d texmax(-DBL_MAX, - DBL_MAX);
  OpenMesh::Vec2d texmin(DBL_MAX,  DBL_MAX);
  for(int i = 0; i < n_u; i++)
  {
    texmax.maximize(OpenMesh::Vec2d(ui[i],vi[i]));
    texmin.minimize(OpenMesh::Vec2d(ui[i],vi[i]));
  }
  double range = (texmax-texmin).max();
  for(TriMesh::HalfedgeIter hi = newmesh.halfedges_begin(); hi != newmesh.halfedges_end(); hi++)
  {
    newmesh.set_texcoord2D(*hi, TriMesh::TexCoord2D(ui[hi->idx()], vi[hi->idx()]) / range * 300 );
  }
  
  emit updatedObject(newID, UPDATE_ALL);
  
  std::cout << "Done." << std::endl;
}

std::vector< std::vector<int> >
SeamlessSimilarityMapsPlugin::
noncontractible_cycle_basis(const TriMesh& m)
{
  std::vector< std::vector<int> > res;
  
  int n_v = m.n_vertices();
  int n_f = m.n_faces();
  int n_e = m.n_edges();
  
  std::vector<bool> etag(n_e, false);
  std::vector<bool> ftag(n_f, false);
  std::vector<bool> vtag(n_v, false);
  
  std::vector<int> dist(n_f, 0);
  std::deque<int> fq;
  
  // construct cut graph
  int cut_root(-1);
  fq.push_front(0);
  ftag[0] = true;
  while(!fq.empty())
  {
    int cur  = fq.back();
    fq.pop_back();
    
    for(TriMesh::FaceHalfedgeIter fh_it = m.cfh_iter(OpenMesh::FaceHandle(cur)); fh_it.is_valid(); ++fh_it)
    {
      int f = m.opposite_face_handle(*fh_it).idx();
      int e = m.edge_handle(*fh_it).idx();
      int v = m.to_vertex_handle(*fh_it).idx();
      
      if(!ftag[f])
      {
        ftag[f] = true;
        dist[f] = dist[cur] + 1;
        fq.push_front(f);
        etag[e] = true;
      }
      else cut_root = v;
    }
  }
  
  // find 2g loops across cut graph bridges
  if(cut_root >= 0)
  {
    // find 2g min-cost bridges across cut graph by Prim's MaxSpanTree
    std::set<OpenMesh::EdgeHandle> bridges;
    
    std::priority_queue< std::pair<int, OpenMesh::HalfedgeHandle> > q;
    vtag[cut_root] = true;
    for(auto voh = m.cvoh_iter(OpenMesh::VertexHandle(cut_root)); voh.is_valid(); ++voh)
    {
      if(etag[m.edge_handle(*voh).idx()]) continue;
      int val = dist[m.face_handle(*voh).idx()] + dist[m.opposite_face_handle(*voh).idx()];
      q.push(std::make_pair(val, *voh));
    }
    while(!q.empty())
    {
      OpenMesh::HalfedgeHandle cur = q.top().second;
      q.pop();
      if(vtag[m.to_vertex_handle(cur).idx()]) { bridges.insert(m.edge_handle(cur)); continue; }
      vtag[m.to_vertex_handle(cur).idx()] = true;
      for(auto voh = m.cvoh_iter(m.to_vertex_handle(cur)); voh.is_valid(); ++voh)
      {
        if(*voh == m.opposite_halfedge_handle(cur)) continue;
        if(etag[m.edge_handle(*voh).idx()]) continue;
        int val = dist[m.face_handle(*voh).idx()] + dist[m.opposite_face_handle(*voh).idx()];
        q.push(std::make_pair(val, *voh));
      }
    }
    
    // connect bridges to root
    for(std::set<OpenMesh::EdgeHandle>::const_iterator b_it = bridges.begin(); b_it != bridges.end(); ++b_it)
    {
      OpenMesh::HalfedgeHandle cur = m.halfedge_handle(*b_it, 0);
      int f1 = m.face_handle(cur).idx();
      int f2 = m.opposite_face_handle(cur).idx();
      OpenMesh::HalfedgeHandle h1 = cur;
      OpenMesh::HalfedgeHandle h2 = m.opposite_halfedge_handle(cur);
      if(dist[f2] > dist[f1]) { std::swap(f1, f2); std::swap(h1, h2); }
      int walker = f1;
      int walkerdist = dist[walker];
      std::vector<int> line;
      if(walkerdist >= 0) line.push_back(h2.idx());
      while(walkerdist >= 1) // walk back to start point
      {
        for(TriMesh::FaceHalfedgeIter fh_it = m.cfh_iter(OpenMesh::FaceHandle(walker)); fh_it.is_valid(); ++fh_it)
        {
          int f = m.opposite_face_handle(*fh_it).idx();
          int e = m.edge_handle(*fh_it).idx();
          if(!etag[e]) continue;
          if(dist[f] == walkerdist-1) { walker = f; walkerdist--; line.push_back(fh_it->idx()); break; }
        }
      }
      walker = f2;
      walkerdist = dist[walker];
      std::vector<int> line2(line.rbegin(), line.rend());
      
      while(walkerdist >= 1) // walk back to start point
      {
        for(TriMesh::FaceHalfedgeIter fh_it = m.cfh_iter(OpenMesh::FaceHandle(walker)); fh_it.is_valid(); ++fh_it)
        {
          int f = m.opposite_face_handle(*fh_it).idx();
          int e = m.edge_handle(*fh_it).idx();
          if(!etag[e]) continue;
          if(dist[f] == walkerdist-1) { walker = f; walkerdist--; line2.push_back(m.opposite_halfedge_handle(*fh_it).idx()); break; }
        }
      }
      while(line2[0] == m.opposite_halfedge_handle(OpenMesh::HalfedgeHandle(line2.back())).idx())
      {
        line2.pop_back();
        line2.erase(line2.begin(),line2.begin()+1);
      }
      res.push_back(line2);
    }
  }
  
  return res;
}


#if QT_VERSION < 0x050000
Q_EXPORT_PLUGIN2( seamlesssimilaritymapsplugin , SeamlessSimilarityMapsPlugin );
#endif

