/**************************************************************
 * This file is part of Deform Factors demo.                  *
 * Project web page:                                          *
 *    http://vcg.isti.cnr.it/deformfactors/                   *
 *                                                            *
 * Copyright (c) 2013 Marco Tarini <marco.tarini@isti.cnr.it> *
 *                                                            *
 * Deform Factors Demo is an implementation of                *
 * the algorithms and data structures described in            *
 * the Scientific Article:                                    *
 *    Accurate and Efficient Lighting for Skinned Models      *
 *    Marco Tarini, Daniele Panozzo, Olga Sorkine-Hornung     *
 *    Computer Graphic Forum, 2014                            *
 *    (presented at EUROGRAPHICS 2014)                        *
 *                                                            *
 * This Source Code is subject to the terms of                *
 * the Mozilla Public License v. 2.0.                         *
 * One copy of the license is available at                    *
 * http://mozilla.org/MPL/2.0/.                               *
 *                                                            *
 * Additionally, this Source Code is CITEWARE:                *
 * any derivative work must cite the                          *
 * above Scientific Article and include the same condition.   *
 *                                                            *
 **************************************************************/


#include <set>
#include <stdio.h>
#include <stdlib.h>


#include "mesh.h"
#include "animation.h"
#include "animation_dqs.h"

#include "QDebug"


const int MAX_TOTAL_BONES = 35;


Mesh::Mesh(){}


void Mesh::computeDeformFactors(){

    std::vector< Vec4 > summator( vert.size(), Vec4(0,0,0,0) );

    for (unsigned int vi=0; vi<vert.size(); vi++){
        for (uint k=0; k<4; k++) {
            vert[vi].deformFactorsTang[k] = 0;
            vert[vi].deformFactorsBtan[k] = 0;

            vert[vi].weightGradient[k].SetZero();
        }
    }

    int nBones = MAX_TOTAL_BONES;

    int overflowCoun = 0;

    for (unsigned int ff=0; ff<face.size(); ff++){
        int pi[3];
        pi[0] = face[ff].index[0];
        pi[1] = face[ff].index[1];
        pi[2] = face[ff].index[2];

        Mat3 m;
        Vec3 e0 = vert[pi[1]].pos - vert[pi[0]].pos;
        Vec3 e1 = vert[pi[2]].pos - vert[pi[0]].pos;
        Vec3 n = e1^e0;

        float faceArea = n.Norm();

        m.SetRow(0, e0 );
        m.SetRow(1, e1 );
        m.SetRow(2, n );
        m = inverse(m);

        std::vector< bool > weightsDone(nBones , false );

        for (int w=0; w<3; w++) {

            for (int r=0; r<4; r++) {

                int bi = vert[ pi[w] ].boneIndex[r];
                if ((bi<0) || (vert[ pi[w] ].boneWeight[r]==0)) continue;
                if ( weightsDone[bi] ) continue;

                weightsDone[bi] = true;

                float dw0 = vert[ pi[1] ].weightOfBone( bi ) - vert[ pi[0] ].weightOfBone( bi );
                float dw1 = vert[ pi[2] ].weightOfBone( bi ) - vert[ pi[0] ].weightOfBone( bi );

                if ((dw0==0)&&(dw1==0)) continue;

                Vec3 faceWeightGradient = ( m * Vec3( dw0, dw1, 0 ) ) ;

                // add it to all verteces
                for (int z=0; z<3; z++) {
                    int k = vert[ pi[z] ].slotOfBone( bi );
                    if (k<0) {
                        overflowCoun++;
                        //vert[ pi[z] ].col = 0xFFFF0000;
                        //vert[ pi[z] ].isTextureFlipped *= 1.0;
                    } else {
                        Vec3 e1 = (vert[ pi[(z+1)%3] ].pos - vert[ pi[z] ].pos);
                        Vec3 e2 = (vert[ pi[(z+2)%3] ].pos - vert[ pi[z] ].pos);
                        float wedgeAngle = angle(e1,e2) * faceArea;
                        vert[ pi[z] ].deformFactorsTang[k] += (vert[ pi[z] ].tang   * faceWeightGradient) * wedgeAngle;
                        vert[ pi[z] ].deformFactorsBtan[k] += (vert[ pi[z] ].bitang * faceWeightGradient) * wedgeAngle;

                        vert[ pi[z] ].weightGradient[k] += faceWeightGradient * wedgeAngle;
                        summator[ pi[z] ][k] += wedgeAngle;
                    }
                }
            }
        }

    }

    for (unsigned int vi=0; vi<vert.size(); vi++){
        for (uint k=0; k<4; k++) {
            if (summator[ vi ][k]) {
                vert[vi].deformFactorsTang[k] /= summator[ vi ][k];
                vert[vi].deformFactorsBtan[k] /= summator[ vi ][k];
                vert[vi].weightGradient[k] /= summator[ vi ][k];
            }
        }
    }


    // shift 1st three
    for (unsigned int vi=0; vi<vert.size(); vi++){
        //vert[vi].orderBoneSlotsWithWeights();
        for (uint k=0; k<3; k++) {
           vert[vi].deformFactorsTang[k] = vert[vi].deformFactorsTang[k+1];
           vert[vi].deformFactorsBtan[k] = vert[vi].deformFactorsBtan[k+1];
        }
    }


    if (overflowCoun) qDebug("%d overflows!",overflowCoun);
}


/* helper function */
int Vert::slotOfBone(int bi){
    for (int i=0; i<4; i++) {
        if (boneIndex[i]==bi) return i;
    }
    for (int i=0; i<4; i++) {
        if (boneIndex[i]==-1) {
            boneIndex[i] = bi;
            boneWeight[i] = 0;
            return i;
        }
    }
    return -1;
}

/* helper function */
float Vert::weightOfBone(int i) const{
  for ( int k=0;  k<4; k++) if (boneIndex[k]==i) return boneWeight[k];
  return 0;
}

void Mesh::computeTangentDirs(){

    // pass 1: reset tang and bitang
    for (unsigned int vi=0; vi<vert.size(); vi++){
        vert[vi].tang=Vec3(0,0,0);
        vert[vi].bitang = Vec3(0,0,0);
    }

    // pass 2: cycle over faces, cumulate tang and bitang
    /* note: we rely on vertex seams, that is, existing vertex duplications
     * in order to store discontinuities of tangent directions.
     */
    for (unsigned int ff=0; ff<face.size(); ff++){
        int vi[3];
        vi[0] = face[ff].index[0];
        vi[1] = face[ff].index[1];
        vi[2] = face[ff].index[2];

        Vec2 s0=vert[ vi[0] ].uv;
        Vec2 s1=vert[ vi[1] ].uv;
        Vec2 s2=vert[ vi[2] ].uv;
        s1-=s0;
        s2-=s0;
        float det = s1^s2;
        if (!det) continue;
        float aT,bT,aB,bB;
        aT = -s2.X()/det;  bT =  s1.X()/det;
        aB =  s2.Y()/det;  bB = -s1.Y()/det;

        Vec3 p0=vert[ vi[0] ].pos;
        Vec3 p1=vert[ vi[1] ].pos;
        Vec3 p2=vert[ vi[2] ].pos;
        p1-=p0;
        p2-=p0;
        float faceArea = (p1^p2).Norm();

        Vec3 faceTangent   = (p1*aT + p2*bT).normalized();
        Vec3 faceBitangent = (p1*aB + p2*bB).normalized();

        for (int z=0; z<3; z++) {
            Vec3 e1 = (vert[ vi[(z+1)%3] ].pos - vert[ vi[z] ].pos);
            Vec3 e2 = (vert[ vi[(z+2)%3] ].pos - vert[ vi[z] ].pos);
            float wedgeAngle = angle(e1,e2) * faceArea;

            vert[ vi[z] ].tang  +=faceTangent   * wedgeAngle;
            vert[ vi[z] ].bitang+=faceBitangent * wedgeAngle;
        }

    }

    // pass 3: for each vertex, make sure tang and bitang are normalized
    //         and orthogonal to normal
    for (unsigned int vi=0; vi<vert.size(); vi++){
        std::swap(vert[vi].tang,vert[vi].bitang);
        vert[vi].bitang =    (  vert[vi].norm ^ vert[vi].bitang ^ vert[vi].norm  ) .Normalize();
        vert[vi].tang   =    (  vert[vi].norm ^ vert[vi].tang   ^ vert[vi].norm  ) .Normalize();
    }
}



void Mesh::computeNormals(){

    // a map from vert positions to normals
    std::map< Vec3, Vec3 > p2n;

    // pass 1: reset tang and bitang
    for (unsigned int vi=0; vi<vert.size(); vi++){
        p2n[ vert[vi].pos ] =Vec3(0,0,0);
    }

    // pass 2: cycle over faces, cumulate norms
    for (unsigned int ff=0; ff<face.size(); ff++){
        int vi[3];
        vi[0] = face[ff].index[0];
        vi[1] = face[ff].index[1];
        vi[2] = face[ff].index[2];

        Vec3 p0=vert[ vi[0] ].pos;
        Vec3 p1=vert[ vi[1] ].pos;
        Vec3 p2=vert[ vi[2] ].pos;
        p1-=p0;
        p2-=p0;

        Vec3 faceNorm= (p2 ^ p1); // this includes area weighting

        for (int z=0; z<3; z++) {
            Vec3 e1 = (vert[ vi[(z+1)%3] ].pos - vert[ vi[z] ].pos);
            Vec3 e2 = (vert[ vi[(z+2)%3] ].pos - vert[ vi[z] ].pos);
            float wedgeAngle = angle(e1,e2);

            p2n[ vert[ vi[z] ].pos ] += faceNorm * wedgeAngle;

        }

    }

    // pass 3: normalize
    for (unsigned int vi=0; vi<vert.size(); vi++){
        vert[vi].norm = p2n[ vert[vi].pos ].normalized();

    }
}


void Mesh::freezeAt(const Pose &p){

    for (uint vi=0; vi<vert.size(); vi++) {
        Vert &v(vert[vi]);

        Vec3 restPos = v.pos;

        v.pos = Vec3(0,0,0);
        for (int k=0; k<MAX_BONES; k++){
            float wieght = v.boneWeight[k];
            int       bi = v.boneIndex [k];
            if (bi>=0 && bi<(int)p.matr.size()) {
                v.pos += (p.matr[bi] * restPos )*wieght;
            }
        }

    }
}

void Mesh::freezeAt(const PoseDQS &p){

    for (uint vi=0; vi<vert.size(); vi++) {
        Vert &v(vert[vi]);

        DualQuaternion d;
        d.a = Quaternion(0,0,0,0);
        d.b = Quaternion(0,0,0,0);

        for (int k=0; k<MAX_BONES; k++){
            float wieght = v.boneWeight[k];
            int       bi = v.boneIndex [k];
            if (bi>=0 && bi<(int)p.quat.size()) {
                d.multiplyAndAdd(p.quat[bi] , wieght );
            }
        }

        d.normalize();

        v.pos = d.applyToPoint( v.pos );

    }
}


void Mesh::removeUnreferencedVertices(){

    // flag used vertices
    std::vector<bool> used( vert.size(), false );
    for (uint i=0; i<face.size(); i++) {
        for (int w=0; w<3; w++) {
            used[ face[i].index[w] ] = true;
        }
    }

    // compress vertex vector
    std::vector<int> remap( vert.size() );
    int j = 0;
    for (uint i=0; i<vert.size(); i++) {
        if (used[i]) {
            vert[j] = vert[i];
            remap[i] = j;
            j++;
        }
    }
    qDebug("From %u to %d vertices",vert.size(),j);
    vert.resize(j);

    // update face-vertex links
    for (uint i=0; i<face.size(); i++) {
        for (int w=0; w<3; w++) {
            face[i].index[w] = remap[ face[i].index[w] ];
        }
    }

}

/* fuse togheter coinciding vertices
 * (SMD format is not indexed!)
 *
 * Leave needed vertex seams, i.e. won't fuse
 * togheter vertices with different
 */
void Mesh::unifyVertices(){

    std::map< Vert, int > v2i;
    for (uint i=0; i<vert.size(); i++) {
        v2i[ vert[i] ] = i;
    }
    for (uint i=0; i<face.size(); i++) {
        for (int w=0; w<3; w++) {
            face[i].index[w] = v2i[ vert[ face[i].index[w] ] ];
        }
    }

    removeUnreferencedVertices();

}


/* vertex ordering, to be used by Mesh::unifyVertices
 * (albitrary lexicographic ordering)
 *
 * Two vertices va and vb will be considered joinable
 * if (!(va<vb) && !(va>vb))
 * that is when va==vb
 */
bool Vert::operator < (const Vert &b) const{

    // if any position or attribute is different, vertex are different
    // (a seam will be left)
    if (pos<b.pos) return true;
    if (pos>b.pos) return false;
    if (uv<b.uv) return true;
    if (uv>b.uv) return false;
    if (norm<b.norm) return true;
    if (norm>b.norm) return false;

    // this is needed to preserve tangent directions:
    // if texture orientation is different, a seam must be left
    if (isTextureFlipped<b.isTextureFlipped) return true;

    // no need: if (isTextureFlipped>b.isTextureFlipped) return false;

    return false;
}

static float sign( float x ){
    return (x<0)?-1:+1;
}

void Mesh::computeIsTextureFlipped(){
    for (unsigned int i=0; i<face.size(); i++) {
        Vec2 e0 = vert[ face[i].index[2] ].uv - vert[ face[i].index[0] ].uv;
        Vec2 e1 = vert[ face[i].index[1] ].uv - vert[ face[i].index[0] ].uv;

        float flipped = sign( e0 ^ e1 );

        for (int w=0; w<3; w++) vert[ face[i].index[w] ].isTextureFlipped = flipped;
    }
}

void Vert::maybeSwapBoneSlots(int i, int j){
    if (boneWeight[i]<boneWeight[j]) {
        std::swap(boneWeight[i],boneWeight[j]);
        std::swap(boneIndex[i],boneIndex[j]);
    }
}


void Vert::orderBoneSlots(){

    maybeSwapBoneSlots( 0, 1 );
    maybeSwapBoneSlots( 2, 3 );
    maybeSwapBoneSlots( 0, 2 );
    maybeSwapBoneSlots( 1, 3 );
    maybeSwapBoneSlots( 1, 2 );

}

void Mesh::orderBoneSlots(){
    for (unsigned int i=0; i<vert.size(); i++)
        vert[i].orderBoneSlots();
}

void Mesh::clear(){
    face.clear();
    vert.clear();
}

bool Mesh::isEmpty() const{
    return (face.size()==0);
}
