/**************************************************************
 * This file is part of Deform Factors demo.                  *
 * Project web page:                                          *
 *    http://vcg.isti.cnr.it/deformfactors/                   *
 *                                                            *
 * Copyright (c) 2013 Marco Tarini <marco.tarini@isti.cnr.it> *
 *                                                            *
 * Deform Factors Demo is an implementation of                *
 * the algorithms and data structures described in            *
 * the Scientific Article:                                    *
 *    Accurate and Efficient Lighting for Skinned Models      *
 *    Marco Tarini, Daniele Panozzo, Olga Sorkine-Hornung     *
 *    Computer Graphic Forum, 2014                            *
 *    (presented at EUROGRAPHICS 2014)                        *
 *                                                            *
 * This Source Code is subject to the terms of                *
 * the Mozilla Public License v. 2.0.                         *
 * One copy of the license is available at                    *
 * http://mozilla.org/MPL/2.0/.                               *
 *                                                            *
 * Additionally, this Source Code is CITEWARE:                *
 * any derivative work must cite the                          *
 * above Scientific Article and include the same condition.   *
 *                                                            *
 **************************************************************/


#include <QtGui>
#include <QShortcut>
#include <QMessageBox>
#include <QPlainTextEdit>
#include <QDialogButtonBox>


#include "mainwindow.h"

QString dataPath;
QString meshPath;
QString animationPath;
QString texturePath;


QString PAPER_TITLE("Accurate and efficient lighting for skinned models");
QString PROJECT_PAGE("http://vcg.isti.cnr.it/deformFactors/");

static int addItemsToComboBoxFromFiles(QComboBox *cb, QString folder, QString subfolder, QString initValue){
    QDir dir(folder+subfolder);
    dir.setNameFilters(QStringList( "*.SMD" ));

    int count = 0;

    QFileInfoList list = dir.entryInfoList();
    for (int i = 0; i < list.size(); ++i) {
        QString val = list.at(i).baseName();
        cb->addItem( val , folder );
        if (QString::compare(val, initValue, Qt::CaseInsensitive)==0) {
            cb->setCurrentIndex( cb->count()-1 );
        }
        count++;
    }
    return count;
}

static void addItemsToComboBoxFromFolder( QComboBox *cb, QString folder) {
    QDir dir(folder);
    dir.setFilter( QDir::Dirs );
    dir.setNameFilters(QStringList( "*" ));
    QFileInfoList list = dir.entryInfoList();
    for (int i = 0; i < list.size(); ++i) {
        QString val = list.at(i).baseName() + "/";
        qDebug("adding: %s",val.toLocal8Bit().data());
        int k = addItemsToComboBoxFromFiles( cb, folder+val, "models/",  "leather_armor_a" );
        qDebug("  (%d entries)",k);
    }
}

void MainWindow::setDataPath(QString newDataPath){
    qDebug("Updating folder to... %s",newDataPath.toLocal8Bit().data());
    dataPath = newDataPath;

    animationPath = newDataPath+"animations/";
    meshPath = newDataPath+"models/";
    texturePath = newDataPath+"textures/";
    ui.animationComboBox->clear();

    addItemsToComboBoxFromFiles( ui.animationComboBox, dataPath, "animations/", "man_walk" );
    ui.glArea->loadAnimation( ui.animationComboBox->currentText() );
}

void MainWindow::selectModelFromCombobox(int newIndex){
    QString newDataPath = ui.modelComboBox->itemData(newIndex).toString();
    if (dataPath == newDataPath) return;
    setDataPath(newDataPath);

}

MainWindow::MainWindow (QWidget * parent):QMainWindow (parent)
{
    setWindowTitle(PAPER_TITLE+" -- DEMO");

    ui.setupUi(this);

#ifdef __APPLE__
    addItemsToComboBoxFromFolder( ui.modelComboBox, QCoreApplication::applicationDirPath() + "/../../../data/");
    setDataPath( QCoreApplication::applicationDirPath() + "/../../../data/humans/" );
#else
    addItemsToComboBoxFromFolder( ui.modelComboBox, "data/");
    setDataPath( "data/humans/" );
#endif
    ui.glArea->loadMesh( ui.modelComboBox->currentText() );


    //ui.glArea->loadAnimation( ui.animationComboBox->currentText() );

    // QT connections

    connect(ui.buttonPlay,   SIGNAL(clicked()), ui.glArea->timer, SLOT(start()) );
    connect(ui.buttonStop,   SIGNAL(clicked()), ui.glArea->timer, SLOT(stop() ) );
    connect(ui.buttonPause,  SIGNAL(clicked()), ui.glArea->timer, SLOT(stop() ) );
    connect(ui.buttonOneStep,SIGNAL(clicked()), ui.buttonPause, SLOT(click()) );
    connect(ui.buttonOneStep,SIGNAL(clicked()), ui.glArea, SLOT(onTimer()) );
    connect(ui.buttonStop,   SIGNAL(clicked()), ui.glArea, SLOT(resetTimer()) );

    connect(ui.buttonShaders, SIGNAL(clicked()), this , SLOT(showShaders() ) );
    connect(ui.buttonAbout,   SIGNAL(clicked()), this , SLOT(aboutWindow() ) );

    connect(ui.modelComboBox, SIGNAL(activated(QString)), ui.glArea, SLOT(loadMesh(QString)) );
    connect(ui.modelComboBox, SIGNAL(currentIndexChanged(int)), this, SLOT(selectModelFromCombobox(int)) );
    connect(ui.animationComboBox, SIGNAL(activated(QString)), ui.glArea, SLOT(loadAnimation(QString)) );

    connect(ui.setFullTrottle, SIGNAL(clicked(bool)), ui.glArea, SLOT(setFullTrottle(bool)) );

    connect(ui.glArea, SIGNAL(signalCurrentFPS(int)), ui.fpsBox, SLOT(display(int)));

    connect( new QShortcut( QKeySequence::Copy  , this ),SIGNAL(activated()), ui.glArea, SLOT(copyViewToClipboard()) );
    connect( new QShortcut( QKeySequence::Paste , this ),SIGNAL(activated()), ui.glArea, SLOT(pasteViewFromClipboard()) );

    connect( ui.buttonExportSnapshots, SIGNAL(clicked()), ui.glArea, SLOT(saveSnapshotSequence()) );
    QButtonGroup * gr1 = new QButtonGroup(this);
    gr1->addButton( ui.setApprox ,         0 );
    gr1->addButton( ui.setCorrect,         1 );
    gr1->addButton( ui.setShowDifferences, 2 );
    gr1->addButton( ui.setShowGroundTruth, 3 );
    connect( gr1 , SIGNAL(buttonClicked(int)), ui.glArea, SLOT( setLightingMode( int ) ) ) ;

    QButtonGroup * gr2 = new QButtonGroup(this);
    gr2->addButton( ui.setLBS , 0 );
    gr2->addButton( ui.setDQS , 1 );
    gr2->addButton( ui.setNoSkinning , 2 );
    connect( gr2 , SIGNAL(buttonClicked(int)), ui.glArea, SLOT( setSkinningMode( int ) ) ) ;

#ifndef MORE_OPTIONS
    ui.buttonExportSnapshots->hide();
    ui.setShowGroundTruth->hide();
    ui.fpsPanel->hide();
#endif

    QFont f = ui.setCorrect->font(); f.setBold(true);
    ui.setCorrect->setFont( f );

    connect( ui.setNoSkinning, SIGNAL(toggled(bool)), ui.lightingPanel, SLOT(setDisabled(bool)) );

}

void MainWindow::aboutWindow(){
    QMessageBox::about(this,"About this demo",
      QString(
        "Demo for the paper<br>"
        "<b>%1</b><br>"
        "Marco Tarini, Daniele Panozzo, Olga Sorkine-Hornung, <br>"
        "Eurographics 2014 (Computer Graphics Forum)<br>"
        "<br>"
        "ver 1.0 (1 dec 2013)<br>"
        "Copyright (C) 2013 Marco Tarini <marco.tarini@isti.cnr.it@gmail.com><br>"
        "<br>"
        "Models, textures and animations from the game<br>"
        "<b>Mount and Blade Warband</b></a><br>"
        "courtesy of <a href=\"http://www.taleworlds.com/\">TaleWorlds Entratainment</a>.<br></i>"
        "<br>"
        ""
        "Additional info, source code, and updates available at the <b>project page</b>:<br>"
        "<a href=\"%2\">%2</a><br>"
        "<br>"
        "<br>"
        "<h3>This software is <b>citeware</b>!</h3>"
        "There is no cost associated to use this software.<br>"
        "However, if you use any animation, screenshot, algorithm, source code "
        "(including GLSL code), from this software, than you need to acknowledge "
        " this by citing the above paper.<br>"
        "If you use it in a webpage or a software, then please also link "
        "the official Project page: <a href=\"%2\">%2</a>"
        "<br>"
       ).arg(PAPER_TITLE).arg(PROJECT_PAGE)
    );
}


QString contentOfFile(QString filename);

void MainWindow::showShaders(){
    QDialog *d = new QDialog(this,Qt::Dialog);

    QFont myfont("Monospace");
    myfont.setStyleHint(QFont::TypeWriter);

    QString vs = ui.glArea->vertexShaderFilename();
    QPlainTextEdit* te0 = new QPlainTextEdit( contentOfFile( vs ),d);
    te0->setFont(myfont);
    te0->setLineWrapMode(QPlainTextEdit::NoWrap);
    te0->setSizePolicy(QSizePolicy::MinimumExpanding, QSizePolicy::MinimumExpanding);

    QString fs = ui.glArea->fragmentShaderFilename();
    QPlainTextEdit* te1 = new QPlainTextEdit( contentOfFile( fs ) ,d);
    te1->setFont(myfont);
    te1->setLineWrapMode(QPlainTextEdit::NoWrap);
    te1->setSizePolicy(QSizePolicy::MinimumExpanding, QSizePolicy::MinimumExpanding);

    QDialogButtonBox* buttonBox = new QDialogButtonBox(QDialogButtonBox::Ok);
    connect(buttonBox, SIGNAL(accepted()), d, SLOT(accept()));

    d->setLayout(new QVBoxLayout(d));
    d->layout()->addWidget( new  QLabel(QString("Vertex shader (%1):").arg(vs),d));
    d->layout()->addWidget( te0 );
    d->layout()->addItem( new QSpacerItem(20, 15, QSizePolicy::Minimum, QSizePolicy::Minimum) );
    d->layout()->addWidget( new  QLabel(QString("Fragment shader (%1):").arg(fs),d));
    d->layout()->addWidget( te1 );
    d->layout()->addItem( new QSpacerItem(20, 15, QSizePolicy::Minimum, QSizePolicy::Minimum) );
    d->layout()->addWidget( buttonBox );

    /*d->setSizePolicy(QSizePolicy::MinimumExpanding, QSizePolicy::Expanding);*/
    d->resize(800,600);

    d->exec();
}


