/**************************************************************
 * This file is part of Deform Factors demo.                  *
 * Project web page:                                          *
 *    http://vcg.isti.cnr.it/deformfactors/                   *
 *                                                            *
 * Copyright (c) 2013 Marco Tarini <marco.tarini@isti.cnr.it> *
 *                                                            *
 * Deform Factors Demo is an implementation of                *
 * the algorithms and data structures described in            *
 * the Scientific Article:                                    *
 *    Accurate and Efficient Lighting for Skinned Models      *
 *    Marco Tarini, Daniele Panozzo, Olga Sorkine-Hornung     *
 *    Computer Graphic Forum, 2014                            *
 *    (presented at EUROGRAPHICS 2014)                        *
 *                                                            *
 * This Source Code is subject to the terms of                *
 * the Mozilla Public License v. 2.0.                         *
 * One copy of the license is available at                    *
 * http://mozilla.org/MPL/2.0/.                               *
 *                                                            *
 * Additionally, this Source Code is CITEWARE:                *
 * any derivative work must cite the                          *
 * above Scientific Article and include the same condition.   *
 *                                                            *
 **************************************************************/

#ifndef GLAREA_H_
#define GLAREA_H_

#include <vector>

#include <QtOpenGL/QGLWidget>
#include <QtOpenGL/QGLShaderProgram>

#include "mesh.h"
#include "animation.h"
#include "animation_dqs.h"

class QFile;
class QTimer;

class GLArea:public QGLWidget
{
    Q_OBJECT
public:
    GLArea (QWidget * parent = 0);


    QTimer *timer, *fpsCounterTimer;

public slots:

    /* load data */
    bool loadMesh(QString filename);
    bool loadBumpMap(QString filename);
    bool loadSpecMap(QString filename);
    bool loadAnimation(QString filename);
    bool loadShaders(QString vertFileName, QString fragFileName );

    void resetTimer();
    void setLightingMode(int mode);
    void setSkinningMode(int mode);
    void setUseNaive(bool mode);
    void setFullTrottle(bool mode);

    void copyViewToClipboard();
    void pasteViewFromClipboard();

    void saveSnapshotSequence();

private slots:
    void onTimer();
    void onFpsTimer();
private:

    /* settings */
    enum { LBS, DQS , NOSKIN } skinningMode;
    enum { CORRECT, APPROX , SHOW_DIFFERENCES, GROUND_TRUTH } lightingMode;
    bool useNaiveLbsCorrect;


    /* opengl initialization and drawing calls */
    void initializeGL ();
    void paintGL ();
    void sendMesh( const Mesh& m );
    void sendPose( const Pose& p );
    void sendPose( const PoseDQS& p );
    void bindAllTextures();
    void reloadShaders();
    static GLuint bindTexture(const QString &fileName);
    static GLuint bindCompressedTexture(const char *buf, int len);


    /* mouse+keyboard callbacks */
    void mousePressEvent(QMouseEvent*e);
    void mouseMoveEvent(QMouseEvent*e);
    void wheelEvent(QWheelEvent*e);
    void keyPressEvent(QKeyEvent *e);

    /* data being shown */
    Mesh currentMesh;
    QString currentBumpmapFilename;
    QString currentSpecmapFilename;
    Animation currentAni;
    AnimationDQS currentAniDqs;
    QString currentNameMesh, currentNameAni;
    int currentFrame;


    /* shaders */
    QGLShaderProgram shaderProgram;
    // location of all vertex attributes and uniforms
    int locOf_pos;
    int locOf_uv;
    int locOf_tang;
    int locOf_norm;
    int locOf_bitang;
    int locOf_boneIndex;
    int locOf_boneWeight;
    int locOf_isTextureFlipped;
    int locOf_deformFactorsTang;
    int locOf_deformFactorsBtan;
    int locOf_weightGradient[MAX_BONES]; /* only used for testing "naive" */
    int locOf_boneMatrices;
    int locOf_boneDualQuaternion;

    /* a small makeshift trackball */
    float eyeDist, eyePhi, eyeTheta;
    QPointF lastMousePos;
    void resetTrackball();
    void setTrackballView();

    /* helper functions, etc */
    FILE* openFile(QFile& f);
    bool meshBuffersReady;
    bool texturesReady;
    void prepareMeshBuffers( const Mesh &m );
    int framesDoneThisSec;
    bool isFullTrottle() const;
public:
    QString vertexShaderFilename() const;
    QString fragmentShaderFilename() const;

signals:
    void signalCurrentFPS(int);

};

#endif /*GLAREA_H_ */
