/**************************************************************
 * This file is part of Deform Factors demo.                  *
 * Project web page:                                          *
 *    http://vcg.isti.cnr.it/deformfactors/                   *
 *                                                            *
 * Copyright (c) 2013 Marco Tarini <marco.tarini@isti.cnr.it> *
 *                                                            *
 * Deform Factors Demo is an implementation of                *
 * the algorithms and data structures described in            *
 * the Scientific Article:                                    *
 *    Accurate and Efficient Lighting for Skinned Models      *
 *    Marco Tarini, Daniele Panozzo, Olga Sorkine-Hornung     *
 *    Computer Graphic Forum, 2014                            *
 *    (presented at EUROGRAPHICS 2014)                        *
 *                                                            *
 * This Source Code is subject to the terms of                *
 * the Mozilla Public License v. 2.0.                         *
 * One copy of the license is available at                    *
 * http://mozilla.org/MPL/2.0/.                               *
 *                                                            *
 * Additionally, this Source Code is CITEWARE:                *
 * any derivative work must cite the                          *
 * above Scientific Article and include the same condition.   *
 *                                                            *
 **************************************************************/

#include <GL/glew.h>
#include <QtOpenGL/QGLWidget>
#include <QApplication>
#include <QTimer>
#include <QFile>
#include <QClipboard>
#include <QMessageBox>
#include <QDir>
#include <QMouseEvent>
#include <QKeyEvent>
#include <QWheelEvent>

#include "glarea.h"
#include "ioSMD.h"

extern QString meshPath;
extern QString animationPath;
extern QString texturePath;
QString shaderPath("shaders/");
QString snapshotPath("screenshots");


#ifdef USE_GLSL_120
#define MY_INDEX_TYPE GL_FLOAT
#else
#define MY_INDEX_TYPE GL_BYTE
#endif

#define ATTR_POINTER_F(NAME) if (locOf_##NAME!=-1) glVertexAttribPointerARB(locOf_##NAME, 1, GL_FLOAT, GL_FALSE, sizeof(Vert), &(m.vert[0]. NAME) );
#define ATTR_POINTER_VF(NAME,NUM) if (locOf_##NAME!=-1) glVertexAttribPointerARB(locOf_##NAME, NUM, GL_FLOAT, GL_FALSE, sizeof(Vert), &(m.vert[0]. NAME[0]) );
#define ATTR_POINTER_VI(NAME,NUM) if (locOf_##NAME!=-1) glVertexAttribIPointer(locOf_##NAME, NUM, MY_INDEX_TYPE, sizeof(Vert), &(m.vert[0]. NAME) );
#define ATTR_POINTER_MF(NAME,NUMA,NUMB) if (locOf_##NAME[0]!=-1) for (int k=0; k<NUMB; k++) glVertexAttribPointerARB(locOf_##NAME[k], NUMA, GL_FLOAT, GL_FALSE, sizeof(Vert), &(m.vert[0].NAME[k][0]) );

void GLArea::prepareMeshBuffers( const Mesh &m ) {

    /* NB: only attributes used in current shader will actually be sent */
    ATTR_POINTER_VF(pos,3 );
    ATTR_POINTER_VF(uv,2);
    ATTR_POINTER_VF(norm,3);
    ATTR_POINTER_VF(tang,3);
    ATTR_POINTER_VF(bitang,3);

#ifdef USE_GLSL_120
    ATTR_POINTER_VF(boneIndex,MAX_BONES);
#else
    ATTR_POINTER_VI(boneIndex,MAX_BONES);
#endif

    ATTR_POINTER_VF(boneWeight,MAX_BONES);

    ATTR_POINTER_VF(deformFactorsTang,MAX_BONES-1);
    ATTR_POINTER_VF(deformFactorsBtan,MAX_BONES-1);
    ATTR_POINTER_F(isTextureFlipped );

    ATTR_POINTER_MF(weightGradient,3,MAX_BONES);

}

void GLArea::sendPose(const Pose &p){
    glUniformMatrix4fvARB(locOf_boneMatrices, p.matr.size(), true, (GLfloat*)&(p.matr[0]) );
}

void GLArea::sendPose(const PoseDQS &p){
    glUniformMatrix2x4fv(locOf_boneDualQuaternion, p.quat.size(), false, (GLfloat*)&(p.quat[0]) );
}

static void maybeEnableVertexAttribArray( int loc ){
    if (loc!=-1) glEnableVertexAttribArrayARB( loc );
}

static void maybeDisableVertexAttribArray( int loc ){
    if (loc!=-1) glDisableVertexAttribArrayARB( loc );
}

void GLArea::sendMesh( const Mesh& m ){
    if (!meshBuffersReady) {
        prepareMeshBuffers( m );
        meshBuffersReady = true;
    }

    maybeEnableVertexAttribArray( locOf_pos );
    maybeEnableVertexAttribArray( locOf_uv );
    maybeEnableVertexAttribArray( locOf_norm );
    maybeEnableVertexAttribArray( locOf_boneWeight );
    maybeEnableVertexAttribArray( locOf_boneIndex );
    maybeEnableVertexAttribArray( locOf_tang );
    maybeEnableVertexAttribArray( locOf_bitang );
    maybeEnableVertexAttribArray( locOf_deformFactorsBtan );
    maybeEnableVertexAttribArray( locOf_deformFactorsTang );
    maybeEnableVertexAttribArray( locOf_isTextureFlipped );
    maybeEnableVertexAttribArray( locOf_weightGradient[0] );
    maybeEnableVertexAttribArray( locOf_weightGradient[1] );
    maybeEnableVertexAttribArray( locOf_weightGradient[2] );
    maybeEnableVertexAttribArray( locOf_weightGradient[3] );


    /* draw geometry (vertex arrays) */
    glDrawElements(GL_TRIANGLES, m.face.size()*3, GL_UNSIGNED_INT, &(m.face[0].index) );

    if (isFullTrottle()) {
        glColorMask( GL_FALSE, GL_FALSE, GL_FALSE, GL_FALSE );
        glDepthMask( GL_FALSE );
        for (int i=0; i<100; i++)
            glDrawElements(GL_TRIANGLES, m.face.size()*3, GL_UNSIGNED_INT, &(m.face[0].index) );
        glColorMask( GL_TRUE, GL_TRUE, GL_TRUE, GL_TRUE );
        glDepthMask( GL_TRUE );
    }

    maybeDisableVertexAttribArray( locOf_pos );
    maybeDisableVertexAttribArray( locOf_uv );
    maybeDisableVertexAttribArray( locOf_norm );
    maybeDisableVertexAttribArray( locOf_boneWeight );
    maybeDisableVertexAttribArray( locOf_boneIndex );
    maybeDisableVertexAttribArray( locOf_tang );
    maybeDisableVertexAttribArray( locOf_bitang );
    maybeDisableVertexAttribArray( locOf_deformFactorsBtan );
    maybeDisableVertexAttribArray( locOf_deformFactorsTang );
    maybeDisableVertexAttribArray( locOf_isTextureFlipped );
    maybeDisableVertexAttribArray( locOf_weightGradient[0] );
    maybeDisableVertexAttribArray( locOf_weightGradient[1] );
    maybeDisableVertexAttribArray( locOf_weightGradient[2] );
    maybeDisableVertexAttribArray( locOf_weightGradient[3] );

}

GLArea::GLArea (QWidget * parent) :QGLWidget (QGLFormat(QGL::SampleBuffers|QGL::AlphaChannel),parent)
{

    resetTrackball();
    timer = new QTimer(this);

    connect(timer,SIGNAL(timeout()),this,SLOT(onTimer()));
    setFullTrottle( false );
    timer->start();

    fpsCounterTimer = new QTimer(this);
    fpsCounterTimer->setInterval(1000);
    connect(fpsCounterTimer,SIGNAL(timeout()),this,SLOT(onFpsTimer()));
    fpsCounterTimer->start();

    currentFrame = 0;
    lightingMode = APPROX;
    skinningMode = LBS;
    texturesReady = false;
    useNaiveLbsCorrect = false;

    #ifdef __APPLE__
        shaderPath = QCoreApplication::applicationDirPath() + "/../../../" + shaderPath;
    #endif

}

void GLArea::setSkinningMode(int mode){
    if (mode==0) skinningMode = LBS;
    else if (mode==1) skinningMode = DQS;
    else if (mode==2) skinningMode = NOSKIN;
    makeCurrent();
    reloadShaders();
    update();
}

void GLArea::setUseNaive(bool mode){
    useNaiveLbsCorrect = mode;
    makeCurrent();
    reloadShaders();
    update();
}



void GLArea::setLightingMode(int mode){
    if (mode==0) lightingMode = APPROX;
    else if (mode==1) lightingMode = CORRECT;
    else if (mode==2) lightingMode = SHOW_DIFFERENCES;
    else if (mode==3) {
        lightingMode = GROUND_TRUTH;
        // for testing comparison, we override the "artist" normals
        // (the ones read in the mesh file)
        // with our "reproducible" normals computed from geometry
        // in the same way
        currentMesh.computeNormals();
        currentMesh.computeTangentDirs();
        currentMesh.computeDeformFactors();
        meshBuffersReady = false;
    }
    makeCurrent();
    reloadShaders();
    update();
}

void GLArea::reloadShaders(){
    loadShaders( vertexShaderFilename(), fragmentShaderFilename() );
    meshBuffersReady = false;
}

void GLArea::onTimer(){
    currentFrame++;
    if ( currentFrame>=(int)currentAni.pose.size() ) currentFrame = 0; // loop ani

    if (isFullTrottle()) updateGL(); // force redraw
    else update();
}

void GLArea::onFpsTimer(){
    emit signalCurrentFPS( framesDoneThisSec );
    framesDoneThisSec = 0;
}

void GLArea::resetTimer(){
    currentFrame = 0;
    update();
}

void GLArea::resetTrackball(){
    eyeDist = 32.0;
    eyePhi = 0.0;
    eyeTheta = 0.0;
}

void GLArea::pasteViewFromClipboard(){
    QString s = QApplication::clipboard()->text();
    QStringList l = s.split(" ");
    if (l.size()!=4) return;
    float f[4];
    bool ok;
    for (int i=0; i<4; i++) {
        f[i] = l.at(i).toFloat(&ok);
        if (!ok) return;
    }
    eyeDist = f[0];
    eyePhi = f[1];
    eyeTheta = f[2];
    currentFrame = int(f[3]);
    if (currentFrame>(int)currentAni.pose.size()) currentFrame = 0;
    update();
}

void GLArea::copyViewToClipboard(){
    QApplication::clipboard()->setText(
       QString("%1 %2 %3 %4").arg(eyeDist).arg(eyePhi).arg(eyeTheta).arg(currentFrame)
    );
}

// substitute gluPerspective (to avoid a dependency)
static void myGluPerspective(GLdouble fovx, GLdouble aspect, GLdouble zNear, GLdouble zFar)
{
   GLdouble xMin, xMax, yMin, yMax;

   xMax = zNear * tan(fovx * M_PI / 360.0);
   xMin = -xMax;

   yMin = xMin / aspect;
   yMax = xMax / aspect;

   GLdouble m[16] = {
       (2.0 * zNear) / (xMax - xMin), 0, 0, 0,

       0, (2.0 * zNear) / (yMax - yMin), 0, 0,

       (xMax + xMin) / (xMax - xMin),
       (yMax + yMin) / (yMax - yMin),
       -(zFar + zNear) / (zFar - zNear),
       -1,

       0, 0, -(2.0 * zFar * zNear) / (zFar - zNear), 0
   };

   glMultMatrixd(m);
}

void GLArea::setTrackballView(){
    // projection matrix
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    int w = width(), h = height();
    glViewport (0, 0, (GLsizei) w, (GLsizei) h);
    myGluPerspective(30, w/(float)h, 0.1, 1000);

    glMatrixMode(GL_MODELVIEW);

    // view matrix
    glLoadIdentity();
    glTranslatef(0,0,-eyeDist);
    glRotatef(eyeTheta, 1,0,0);
    glRotatef(eyePhi, 0,1,0);

    // model matrix
    // quick hack: models center is about point 0,10,0
    // todo: compute bounding box instead
    glTranslatef(0,-10,0);

}

// small helper functions

bool tryOpen(QFile& f){
    if (!f.open(QIODevice::ReadOnly)) {
        QMessageBox::warning(
            NULL,"Deform Weights",
            QString("cannot open file '%1'").arg(f.fileName())
        );
        return false;
    }
    return true;
}

QString contentOfFile(QString filename){
    QFile fa( filename );
    if (!tryOpen(fa)) return QString();
    else return QString( fa.readAll() );
}

FILE* GLArea::openFile(QFile &qfile){
    if (!tryOpen(qfile)) return NULL;
    return fdopen(qfile.handle(),"rt");
}


void hackForceToGLSL120(QString &s){
    s.replace("#version 130", "#version 120");
    s.replace("#version 150", "#version 120");
    s.replace("attribute ivec", "attribute  vec");
    s.replace("boneIndex[0]", "int(boneIndex[0])");
    s.replace("boneIndex[1]", "int(boneIndex[1])");
    s.replace("boneIndex[2]", "int(boneIndex[2])");
    s.replace("boneIndex[3]", "int(boneIndex[3])");
}

void GLArea::saveSnapshotSequence(){
    QDir folder(snapshotPath);
    QString subfolderName = QString("%1_%2").arg(currentNameMesh).arg(currentNameAni);
    folder.mkdir(subfolderName);


    QString skinModeName;
    switch(skinningMode){
    case LBS: skinModeName = "LBS"; break;
    case DQS: skinModeName = "DQS"; break;
    case NOSKIN: skinModeName = "NOS"; break;
    }

    QString techName;
    switch(lightingMode){
    case CORRECT: techName = "ours"; break;
    case APPROX: techName = "trad"; break;
    case SHOW_DIFFERENCES: techName = "diff"; break;
    case GROUND_TRUTH: techName = "true"; break;
    }

    if ((skinningMode==LBS) && (useNaiveLbsCorrect) && (lightingMode==CORRECT) ) techName = "naiv";

    for (currentFrame=0; currentFrame<(int)currentAni.pose.size(); currentFrame++) {

        QString filename = QString("%1/%2/%3")
                .arg(snapshotPath)
                .arg(subfolderName)
                .arg( skinModeName );
        if (skinningMode!=NOSKIN) filename += QString("_%1_%2")
                .arg( techName )
                .arg(currentFrame, 3, 10, QChar('0') );

        bool useTransparency = false;
#ifdef TRANSPARENT_SNAPSHOTS
        useTransparency = true;
#endif

        updateGL();
        glFlush();
        swapBuffers();
        grabFrameBuffer( useTransparency ).save(filename+".png");

        if (skinningMode==NOSKIN) break;
    }
}

bool GLArea::loadShaders(QString vertFileName, QString fragFileName){

    shaderProgram.removeAllShaders();
#ifdef USE_GLSL_120
    QString sourceCode = contentOfFile(vertFileName) ;
    hackForceToGLSL120( sourceCode );
    shaderProgram.addShaderFromSourceCode( QGLShader::Vertex,   sourceCode );
#else
    shaderProgram.addShaderFromSourceFile( QGLShader::Vertex,   vertFileName );
#endif

    shaderProgram.addShaderFromSourceFile( QGLShader::Fragment, fragFileName );
    shaderProgram.link();

    shaderProgram.bind();

    /* find locations of all possible vertex attributes...
     * the ones which aren't used by current shader will be -1 */

    locOf_norm = shaderProgram.attributeLocation("norm");
    locOf_pos = shaderProgram.attributeLocation("pos");
    locOf_uv = shaderProgram.attributeLocation("uv");
    locOf_tang = shaderProgram.attributeLocation("tang");
    locOf_bitang = shaderProgram.attributeLocation("bitang");
    locOf_boneIndex = shaderProgram.attributeLocation("boneIndex");
    locOf_boneWeight = shaderProgram.attributeLocation("boneWeight");;
    locOf_deformFactorsTang = shaderProgram.attributeLocation("deformFactorsTang");
    locOf_deformFactorsBtan = shaderProgram.attributeLocation("deformFactorsBitang");
    locOf_isTextureFlipped = shaderProgram.attributeLocation("isTextureFlipped");

    locOf_weightGradient[0] = shaderProgram.attributeLocation("weightGradient0");
    locOf_weightGradient[1] = shaderProgram.attributeLocation("weightGradient1");
    locOf_weightGradient[2] = shaderProgram.attributeLocation("weightGradient2");
    locOf_weightGradient[3] = shaderProgram.attributeLocation("weightGradient3");
    locOf_boneMatrices = shaderProgram.uniformLocation("boneMatrix");
    locOf_boneDualQuaternion = shaderProgram.uniformLocation("boneDualQuaternion");

    shaderProgram.setUniformValue("samplerBump",0);
    shaderProgram.setUniformValue("samplerSpec",1);

    meshBuffersReady = false;

    return true;

}

QString GLArea::fragmentShaderFilename() const{

    return shaderPath + ( (lightingMode==SHOW_DIFFERENCES)?"differences":"common" )+".fragment.glsl";
}

QString GLArea::vertexShaderFilename() const{    

    if ( (lightingMode==GROUND_TRUTH) || (skinningMode==NOSKIN) )
    return shaderPath + QString("no_skinning.vertex.glsl");

    QString modeName = ( (lightingMode==APPROX)?"old_way":"new_way" );
    QString skinName = ( (skinningMode==DQS)?"DQS":"LBS" );
    if (useNaiveLbsCorrect && (skinningMode==LBS) && (lightingMode==CORRECT)) modeName = "naive_way";
    return shaderPath + QString("%2_%1.vertex.glsl").arg( skinName  ).arg( modeName );
}

void GLArea::initializeGL()
{
    glewInit();
    glClearColor(1, 1, 1, 0);
    glEnable(GL_DEPTH_TEST);
    glPolygonOffset(1,1);
    reloadShaders();
    glEnable(GL_MULTISAMPLE);
    glEnable(GL_CULL_FACE);
    glFrontFace(GL_CW);

    QString res;
    qDebug(res.toLocal8Bit().data());

}

bool GLArea::loadBumpMap(QString filename){
    currentBumpmapFilename = filename;
    update();
    return true;
}

bool GLArea::loadSpecMap(QString filename){
    currentSpecmapFilename = filename;
    update();
    return true;
}

void GLArea::setFullTrottle(bool mode){
    if (mode) {
        timer->setInterval(0);
        timer->start();
    }
    else {
        timer->setInterval(1000/20);
    }
}

bool GLArea::isFullTrottle() const{
    return (timer->interval()==0);
}

bool GLArea::loadMesh(QString filename){

    qDebug("Loading mesh %s",filename.toLocal8Bit().data());
    currentNameMesh = filename;
    QFile f(meshPath+filename+".SMD");

    Animation tmpAni;
    if (!ioSMD::import( openFile( f ), currentMesh, tmpAni )) {
        qDebug("Loading mesh: %s",ioSMD::lastErrorString() );
        return false;
    }


    if (!tmpAni.isEmpty()) {
        // if the mesh file also embeds an animation???
    }


    currentMesh.orderBoneSlots();
    currentMesh.computeIsTextureFlipped();
    currentMesh.unifyVertices(); // comment this line for flat shading!
    currentMesh.computeTangentDirs();
    currentMesh.computeDeformFactors();

    meshBuffersReady = false;
    texturesReady = false;

    QString basename = filename;

    currentBumpmapFilename = texturePath + basename + "_normalmap.dds";
    currentSpecmapFilename = texturePath + basename + "_specular.dds";

    update();
    return true;
}



bool GLArea::loadAnimation(QString filename){
    qDebug("Loading animation %s",filename.toLocal8Bit().data());

    currentNameAni = filename;

    QFile f(animationPath+filename+".SMD");

    Mesh tmpMesh;
    if (!ioSMD::import(  openFile( f ), tmpMesh, currentAni ))  qDebug("ERROR: %s",ioSMD::lastErrorString() );

    // small hack: in our looped animations, last frame == 1st frame, so we remove it
    currentAni.pose.pop_back();

    currentAniDqs.buildFromAnimation( currentAni );
    currentFrame = 0;

    update();

    return true;
}


// DDS format structure
struct DDSFormat {
    quint32 dwSize;
    quint32 dwFlags;
    quint32 dwHeight;
    quint32 dwWidth;
    quint32 dwLinearSize;
    quint32 dummy1;
    quint32 dwMipMapCount;
    quint32 dummy2[11];
    struct {
        quint32 dummy3[2];
        quint32 dwFourCC;
        quint32 dummy4[5];
    } ddsPixelFormat;

    enum {
        FOURCC_DXT1 = 0x31545844,
        FOURCC_DXT2 = 0x32545844,
        FOURCC_DXT3 = 0x33545844,
        FOURCC_DXT4 = 0x34545844,
        FOURCC_DXT5 = 0x35545844
    };

};



GLuint GLArea::bindCompressedTexture(const char *buf, int len)
{
    if (len < 4 || qstrncmp(buf, "DDS ", 4)) return 0;

    // TODO: bail out if the necessary extension is not present.

    const DDSFormat *ddsHeader = reinterpret_cast<const DDSFormat *>(buf + 4);
    if (!ddsHeader->dwLinearSize) {
        qWarning("QGLContext::bindTexture(): DDS image size is not valid.");
        return 0;
    }

    int blockSize = 16;
    GLenum format;

    switch(ddsHeader->ddsPixelFormat.dwFourCC) {
    case DDSFormat::FOURCC_DXT1:
        format = GL_COMPRESSED_RGBA_S3TC_DXT1_EXT;
        blockSize = 8;
        break;
    case DDSFormat::FOURCC_DXT3:
        format = GL_COMPRESSED_RGBA_S3TC_DXT3_EXT;
        break;
    case DDSFormat::FOURCC_DXT5:
        format = GL_COMPRESSED_RGBA_S3TC_DXT5_EXT;
        break;
    default:
        qWarning("QGLContext::bindTexture(): DDS image format not supported.");
        return 0;
    }

    const GLubyte *pixels =
        reinterpret_cast<const GLubyte *>(buf + ddsHeader->dwSize + 4);

    unsigned int id;
    glGenTextures(1, &id);
    glBindTexture(GL_TEXTURE_2D, id);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);

    int size;
    int offset = 0;
    int available = len - int(ddsHeader->dwSize + 4);
    int w = ddsHeader->dwWidth;
    int h = ddsHeader->dwHeight;

    // load mip-maps
    for(int i = 0; i < (int) ddsHeader->dwMipMapCount; ++i) {
        if (w == 0) w = 1;
        if (h == 0) h = 1;

        size = ((w+3)/4) * ((h+3)/4) * blockSize;
        if (size > available) break;

        glCompressedTexImage2DARB(GL_TEXTURE_2D, i, format, w, h, 0, size, pixels + offset);
        offset += size;
        available -= size;

        // half size for each mip-map level
        w = w/2;
        h = h/2;
    }

    return id;
}


/* redefining this method because Qt's
 * "QGLWidget::bindTexture" is a dick, refursing to work if
 * GL_EXT_texture_compression_s3tc is not present
 * (when GL_ARB_texture_compression would be enough)
 */
GLuint GLArea::bindTexture(const QString &fileName)
{
    QFile file(fileName);
    if (!file.open(QIODevice::ReadOnly)) return 0;
    QByteArray contents = file.readAll();
    file.close();

    return bindCompressedTexture(contents.constData(), contents.size());
}

void GLArea::bindAllTextures(){

    if (texturesReady) return;
    glActiveTextureARB(GL_TEXTURE0_ARB);
    if (!bindTexture(currentBumpmapFilename))
    {
        if (!bindTexture(texturePath+"default_normalmap.dds") )
        {
            qDebug("Cannot load norm texture!");
        }
    }
    glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR );
    glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR );


    glActiveTextureARB(GL_TEXTURE1_ARB);
    if (!bindTexture(currentSpecmapFilename))
    {
        qDebug("Cannot load spec texture! '%s'",currentSpecmapFilename.toLocal8Bit().data());
        if (!bindTexture(texturePath+"dafault_specular.dds") )
        {
            qDebug("Cannot load spec texture! '%s'",(texturePath+"default_specular.dds").toLocal8Bit().data());
        }
    }
    glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR );
    glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR );

    texturesReady = true;
}


void GLArea::paintGL()
{

    glClear(GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT);

    setTrackballView();

    bindAllTextures();
    shaderProgram.bind();

    if ( (lightingMode==GROUND_TRUTH) && (skinningMode!=NOSKIN) ) {

        // CPU skinning

        Mesh tmpMesh = currentMesh;
        if (skinningMode == LBS )
            tmpMesh.freezeAt( currentAni.pose[ currentFrame ] );
        else
            tmpMesh.freezeAt( currentAniDqs.pose[ currentFrame ] );
        tmpMesh.computeNormals();
        tmpMesh.computeTangentDirs();
        meshBuffersReady = false;
        sendMesh( tmpMesh );
        meshBuffersReady = false;
    } else {

        // GPU skinning

        if (skinningMode == LBS ) {
          sendPose( currentAni.pose[ currentFrame ] );
        } else {
          sendPose( currentAniDqs.pose[ currentFrame ] );
        }
        sendMesh( currentMesh );
    }

    framesDoneThisSec++;

} 

void GLArea::mousePressEvent(QMouseEvent * e)
{
    lastMousePos = e->pos();
    e->accept();
    setFocus();
    updateGL();
}

void GLArea::mouseMoveEvent(QMouseEvent * e)
{
    QPointF dp = e->pos() - lastMousePos;
    lastMousePos = e->pos();

    eyePhi += dp.x();
    eyeTheta += dp.y();
    if (eyeTheta>+90) eyeTheta=+90;
    if (eyeTheta<-90) eyeTheta=-90;
    updateGL();
}

void GLArea::wheelEvent(QWheelEvent * e)
{
    /* wheel ==> zoom */
    if (e->delta()>0) eyeDist *= 1.1;
    else eyeDist /= 1.1;
    if (eyeDist>150) eyeDist=150;
    if (eyeDist<10) eyeDist=10;
    updateGL();
}

void GLArea::keyPressEvent(QKeyEvent *e){
    if (e->key() == Qt::Key_R ) {
        makeCurrent();
        reloadShaders();
        update();
    }

    /* hot key: N to set naive comparison */
    if (e->key() == Qt::Key_N ) {
        if ((skinningMode==LBS) && (lightingMode==CORRECT) ) setUseNaive( !useNaiveLbsCorrect );
    }
}

